#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Fixture;
use Test::Partner::Utils qw(use_common_blocks_table);

use Test::More;
use Test::Deep;

use qbit;

use lib::abs;

my @media_dsp_fixtures = (
    qw(
      dsp_media
      dsp_media_video_mobile
      dsp_media_video_mobile_with_internal_tag
      )
);

my @dsp_fixtures = (
    @media_dsp_fixtures,
    qw(
      dsp_direct
      dsp_video
      dsp_mobile
      )
);

my @methods            = (qw(get_available_dsps get_default_dsps));
my @mobile_block_types = (qw(banner interstitial native));

# TODO: internal blocks, private_auction

my @block_fixtures = (
    'partner_context_rtb_block',                   'partner_context_rtb_vip_block',
    'adinside_context_rtb_block',                  'business_unit_context_rtb_block',
    'internal_context_rtb_block',                  'partner_mobile_native_rtb_block',
    'partner_mobile_banner_rtb_block',             'partner_mobile_interstitial_rtb_block',
    'partner_mobile_rewarded_rtb_block',           'adinside_mobile_interstitial_rtb_block',
    'business_unit_mobile_interstitial_rtb_block', 'partner_instream_video_block',
    'partner_inpage_video_block',                  'partner_fullscreen_video_block',
    'partner_outdoor_video_block',
);

my %turn_off_first = map {$_ => 1} (
    qw(
      dsp_direct
      )
);

my %block_name_by_id;

run_tests(
    sub {
        my ($app) = @_;

        my $dsps       = get_fixtures_hash(@dsp_fixtures);
        my $media_dsps = get_fixtures_hash(@media_dsp_fixtures);

        $app->set_cur_user(get_fixture('mocked_dsp_manager'));

        for my $block_name (@block_fixtures) {
            my $block_id = get_normalized_block_id(get_fixture($block_name));
            $block_name_by_id{$block_id->{page_id}}{$block_id->{id}} = $block_name;
        }

        use_common_blocks_table($app);

        my $file = lib::abs::path('turn_on_dsp.json');

        unless (-e $file) {
            writefile($file, '{}');
        }

        my $expected_data = from_json(readfile($file));
        my $got_data      = {};

        for my $dsp_name (keys(%$dsps)) {
            my $dsp = $dsps->{$dsp_name};
            if ($turn_off_first{$dsp_name}) {
                $app->dsp->do_action($dsp->{id}, 'stop_on_all_platforms');
            }
            $app->dsp->do_action($dsp->{id}, 'start_on_all_platforms');
            $app->dsp->do_action($dsp->{id}, 'start_on_yandex_services') if $media_dsps->{$dsp_name};
            my $links = get_all_dsp_links($app, $dsp);
            cmp_deeply($links, bag(@{$expected_data->{$dsp_name}}), $dsp_name)
              unless need_self_update();
            $got_data->{$dsp_name} = [sort @$links];
        }

        if (need_self_update()) {
            writefile($file, to_json($got_data, pretty => TRUE));
            fail('self update mode');
        }
    },
    fill_databases => 0,
    fixtures       => [@block_fixtures, @dsp_fixtures,],
);

sub get_all_dsp_links {
    my ($app, $dsp) = @_;

    my @links = map {
        my $id = get_normalized_block_id($_);
        $block_name_by_id{$id->{page_id}}{$id->{id}}
      } @{
        $app->partner_db->block_dsps->get_all(
            fields => [qw(page_id block_id dsp_id)],
            filter => [dsp_id => '=' => \$dsp->{id}]
        )
      };
    return \@links;
}

sub get_normalized_block_id {
    my ($block) = @_;

    my $id = {};
    for my $possible_page_id_name (qw(page_id campaign_id context_page_id)) {
        $id->{page_id} = $block->{$possible_page_id_name} if exists($block->{$possible_page_id_name});
    }

    for my $possible_block_id_name (qw(block_id id)) {
        $id->{id} = $block->{$possible_block_id_name} if exists($block->{$possible_block_id_name});
    }

    return $id;
}
