use strict;
use warnings FATAL => 'all';
use utf8;

use Test::Partner2::Simple;
use Test::More;
use Test::Deep qw(cmp_deeply ignore);
use Test::Exception;
use qbit;

sub check_successful_add {
    my ($app) = @_;

    change_cur_user($app, 'mocked-yan-partner');

    my $page_id = 41443;
    my $tag_id  = 8;

    my $ids = $app->block_tags->add(
        page_id => $page_id,
        tag_id  => $tag_id,
        caption => 'aaa',
    );
    cmp_deeply($ids, $tag_id . '_' . $page_id, 'got expected ids after add()');

    cmp_deeply(
        [map {$_->{opts} = from_json($_->{opts}); $_;} @{$app->partner_db->block_tags_action_log->get_all()}],
        [
            {
                action         => 'add',
                comment        => '',
                dt             => ignore(),
                elem_page_id   => $page_id,
                elem_tag_id    => $tag_id,
                id             => 1,
                new_multistate => 1,
                old_multistate => 0,
                opts           => {
                    "caption"   => "aaa",
                    "page_id"   => $page_id,
                    "public_id" => "${tag_id}_${page_id}",
                    "tag_id"    => $tag_id,
                },
                user_id => 1009,
            },
        ],
        'got expected data in block_tags_action_log',
    );
}

sub check_unsuccessful_add_duplicate {
    my ($app) = @_;

    change_cur_user($app, 'mocked-yan-partner');

    my $page_id = 41443;
    my $tag_id  = 1;

    eval {my $ids = $app->block_tags->add(page_id => $page_id, tag_id => $tag_id, caption => 'bbb',);};
    is(
        $@->message(),
        sprintf('Caption for Page ID %s Tag ID %s already exists.', $page_id, $tag_id),
        'add() throws exception on duplicate ',
      );

}

sub check_unsuccessful_add_incorrect_data {
    my ($app) = @_;

    change_cur_user($app, 'mocked-yan-partner');

    throws_ok(
        sub {
            my $ids = $app->block_tags->add(page_id => 'some string', caption => 'x' x 256);
        },
        'Exception::Validator::Errors',
        'add() throws exception on incorrect data',
    );

    # test add incorrect data to page_id

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => -41443,
                tag_id  => 2,
                caption => 'ddd',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if negative value  in page_id',
    );

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => 41443.5,
                tag_id  => 2,
                caption => 'ddd',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if float value in page_id',
    );

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => 'page_id',    # for int_un '13' is ok
                tag_id  => 2,
                caption => 'ddd',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if string value in page_id',
    );

    # test add incorrect data to page_id

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => '41443',
                tag_id  => 1_000_000_000 + 1,
                caption => 'ddd',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if tag_id value over than max value',
    );

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => '41443',
                tag_id  => 'tag_id',
                caption => 'ddd',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if string value in tag_id',
    );

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => '41443',
                tag_id  => 1.5,
                caption => 'ddd',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if float value in tag_id',
    );

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => '41443',
                tag_id  => 'tag_id',
                caption => 'ddd',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if string value in tag_id',
    );

    # test add incorrect data to caption

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => '41443',
                tag_id  => '2',
                caption => '',
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if empty string value in caption',
    );

    throws_ok(
        sub {
            $app->block_tags->add(
                page_id => '41443',
                tag_id  => '2',
                caption => 'x' x 256,
              ),

        },
        'Exception::Validator::Errors',
        'add() throws exception if string len over than caption max len',
    );

}

run_tests(
    sub {
        my ($app) = @_;

        check_successful_add($app);
        check_unsuccessful_add_duplicate($app);
        check_unsuccessful_add_incorrect_data($app);

    },
);
