#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::More;
use Test::Deep;

use qbit;

my $fixture = {
    'new' => {
        'first add' => {
            args => {
                cur_pos  => 0,
                dest_pos => undef,
                max_pos  => 0,
            },
            expect => {
                new_pos => 1,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },
        'to end' => {
            args => {
                cur_pos  => 0,
                dest_pos => undef,
                max_pos  => 1,
            },
            expect => {
                new_pos => 2,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },
        'to end 2' => {
            args => {
                cur_pos  => 0,
                dest_pos => undef,
                max_pos  => 2,
            },
            expect => {
                new_pos => 3,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },
        'to archive' => {
            args => {
                cur_pos  => 2,
                dest_pos => undef,
                max_pos  => 5,
            },
            expect => {
                new_pos => 5,
                sign    => -1,
                from    => 3,
                to      => 5,
            }
        },
        'to archive last 1' => {
            args => {
                cur_pos  => 7,
                dest_pos => undef,
                max_pos  => 5,
            },
            expect => {
                new_pos => undef,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },
        'to archive last 2' => {
            args => {
                cur_pos  => 5,
                dest_pos => undef,
                max_pos  => 5,
            },
            expect => {
                new_pos => 5,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },
        'to archive last active' => {
            args => {
                cur_pos  => 1,
                dest_pos => undef,
                max_pos  => undef,
            },
            expect => {
                new_pos => undef,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },
        'delete stopped with no active rules' => {
            args => {
                cur_pos  => 0,
                dest_pos => undef,
                max_pos  => undef,
            },
            expect => {
                new_pos => undef,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },
        'before last 1' => {
            args => {
                cur_pos  => 0,
                dest_pos => 1,
                max_pos  => 1,
            },
            expect => {
                new_pos => 1,
                sign    => 1,
                from    => 1,
                to      => 1,
            }
        },
        'before last 2' => {
            args => {
                cur_pos  => 0,
                dest_pos => 1,
                max_pos  => 2,
            },
            expect => {
                new_pos => 1,
                sign    => 1,
                from    => 1,
                to      => 2,
            }
        },
    },
    'existsing' => {

        # nothing changed
        'nothing changed. last to first' => {
            args => {
                cur_pos  => 1,
                dest_pos => 1,
                max_pos  => undef,
            },
            expect => {
                new_pos => 1,
                sign    => 0,
                from    => 0,
                to      => 0,
            }
        },

        # backward
        'backward. last to first' => {
            args => {
                cur_pos  => 3,
                dest_pos => 1,
                max_pos  => undef,
            },
            expect => {
                new_pos => 1,
                sign    => 1,
                from    => 1,
                to      => 2,
            }
        },
        'backward. middle to first' => {
            args => {
                cur_pos  => 3,
                dest_pos => 1,
                max_pos  => undef,
            },
            expect => {
                new_pos => 1,
                sign    => 1,
                from    => 1,
                to      => 2,
            }
        },
        'backward. middle to second' => {
            args => {
                cur_pos  => 3,
                dest_pos => 2,
                max_pos  => undef,
            },
            expect => {
                new_pos => 2,
                sign    => 1,
                from    => 2,
                to      => 2,
            }
        },
        'backward. last before middle' => {
            args => {
                cur_pos  => 7,
                dest_pos => 3,
                max_pos  => undef,
            },
            expect => {
                new_pos => 3,
                sign    => 1,
                from    => 3,
                to      => 6,
            }
        },

        # forward
        'forward. first to end' => {
            args => {
                cur_pos  => 1,
                dest_pos => undef,
                max_pos  => 3,
            },
            expect => {
                new_pos => 3,
                sign    => -1,
                from    => 2,
                to      => 3,
            }
        },
        'forward. first to last' => {
            args => {
                cur_pos  => 1,
                dest_pos => 3,
                max_pos  => undef,
            },
            expect => {
                new_pos => 3,
                sign    => -1,
                from    => 2,
                to      => 3,
            }
        },
        'forward. middle to the end' => {
            args => {
                cur_pos  => 2,
                dest_pos => undef,
                max_pos  => 4,
            },
            expect => {
                new_pos => 4,
                sign    => -1,
                from    => 3,
                to      => 4,
            }
        },
        'forward. middle to last' => {
            args => {
                cur_pos  => 2,
                dest_pos => 4,
                max_pos  => undef,
            },
            expect => {
                new_pos => 4,
                sign    => -1,
                from    => 3,
                to      => 4,
            }
        },
        'forward. middle before last 4' => {
            args => {
                cur_pos  => 2,
                dest_pos => 3,
                max_pos  => undef,
            },
            expect => {
                new_pos => 3,
                sign    => -1,
                from    => 3,
                to      => 3,
            }
        },
        'forward. middle before last 5' => {
            args => {
                cur_pos  => 2,
                dest_pos => 4,
                max_pos  => undef,
            },
            expect => {
                new_pos => 4,
                sign    => -1,
                from    => 3,
                to      => 4,
            }
        },
    }
};

run_tests(
    sub {
        my ($app) = @_;

        foreach my $test_name (sort keys %$fixture) {

            subtest $test_name => sub {

                foreach my $sub_test_name (sort keys %{$fixture->{$test_name}}) {

                    my $test_data = $fixture->{$test_name}->{$sub_test_name};

                    my ($cur_pos, $dest_pos, $max_pos) = @{$test_data->{args}}{qw( cur_pos  dest_pos  max_pos )};

                    my ($new_pos, $sign, $from, $to) =
                      $app->business_rules->_get_move_params($cur_pos, $dest_pos, $max_pos);
                    my $got = {
                        new_pos => $new_pos,
                        sign    => $sign,
                        from    => $from,
                        to      => $to,
                    };

                    my $expect = $test_data->{expect};

                    my $is_ok = cmp_deeply($got, $expect, $test_name . '. ' . $sub_test_name);
                    warn Data::Dumper->Dump([$got], ['$got ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;
                }
            };
        }
    },
    dont_create_database => TRUE
);
