#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::More;
use Test::Differences qw( eq_or_diff );

use qbit;

my $fixture = [
    {
        test_name         => 'Edit stopped',
        action            => 'edit',
        rule_id           => 12,
        args              => {blocks => [qw( R-A-41443-100000 )]},
        expect_multistate => 'Stopped',
        expect            => [
            {
                block_public_id => 'D-A-41443-100000',
                rule_id         => '10',
                is_deleted      => 1,
            },
            {
                block_public_id => 'R-A-41443-100000',
                rule_id         => '12',
                is_deleted      => 0,
            },
        ]
    },
    {
        test_name         => 'Start stopped',
        action            => 'start',
        rule_id           => 12,
        args              => undef,
        expect_multistate => 'Working',
        expect            => [
            {
                rule_id         => '10',
                is_deleted      => '1',
                block_public_id => 'D-A-41443-100000'
            },
            {
                rule_id         => '12',
                is_deleted      => '0',
                block_public_id => 'R-A-41443-100000'
            }
        ]
    },
    {
        test_name         => 'Move working',
        action            => 'move',
        rule_id           => 12,
        args              => {insert_before => 10},
        expect_multistate => 'Working',
        expect            => [
            {
                rule_id         => '10',
                is_deleted      => '1',
                block_public_id => 'D-A-41443-100000'
            },
            {
                rule_id         => '12',
                is_deleted      => '0',
                block_public_id => 'R-A-41443-100000'
            }
        ]
    },
    {
        test_name         => 'Stop working',
        action            => 'stop',
        rule_id           => 12,
        args              => undef,
        expect_multistate => 'Stopped',
        expect            => [
            {
                rule_id         => '10',
                is_deleted      => '1',
                block_public_id => 'D-A-41443-100000'
            },
            {
                rule_id         => '12',
                is_deleted      => '0',
                block_public_id => 'R-A-41443-100000'
            }
        ]
    },
    {
        test_name         => 'Delete stopped',
        action            => 'delete',
        rule_id           => 12,
        args              => undef,
        expect_multistate => 'Stopped. Archived',
        # TODO: можно и не посылать, для БК ничего не меняется
        expect => [
            {
                rule_id         => '10',
                is_deleted      => '1',
                block_public_id => 'D-A-41443-100000'
            },
            {
                rule_id         => '12',
                is_deleted      => '0',
                block_public_id => 'R-A-41443-100000'
            }
        ]
    },
    {
        test_name         => 'Restore deleted',
        action            => 'restore',
        rule_id           => 12,
        args              => undef,
        expect_multistate => 'Stopped',
        expect            => []
    },
    {
        test_name         => 'Start stopped #2',
        action            => 'start',
        rule_id           => 12,
        args              => undef,
        expect_multistate => 'Working',
        expect            => [
            {
                rule_id         => '10',
                is_deleted      => '1',
                block_public_id => 'D-A-41443-100000'
            },
            {
                rule_id         => '12',
                is_deleted      => '0',
                block_public_id => 'R-A-41443-100000'
            }
        ]
    },
    {
        test_name         => 'Delete working',
        action            => 'delete',
        rule_id           => 12,
        args              => undef,
        expect_multistate => 'Stopped. Archived',
        expect            => [
            {
                rule_id         => '10',
                is_deleted      => '1',
                block_public_id => 'D-A-41443-100000'
            },
            {
                rule_id         => '12',
                is_deleted      => '0',
                block_public_id => 'R-A-41443-100000'
            }
        ]
    },
    {
        test_name => 'Add',
        action    => 'add',
        args      => {
            login      => "mocked-yan-partner",
            caption    => "XXX",
            cpm        => 9999,
            conditions => {},
            blocks     => [qw( R-A-142898-1 )]
        },
        expect_multistate => 'Stopped',
        # TODO: можно и не посылать, для БК ничего не меняется
        expect => [
            {
                rule_id         => '21',            # <--!!
                is_deleted      => '1',
                block_public_id => 'R-A-142898-1'
            },
            {
                rule_id         => '22',
                is_deleted      => '0',
                block_public_id => 'R-A-142898-1'
            }
        ]
    },
];

run_tests(
    sub {
        my ($app) = @_;

        my $counter = 3;
        foreach my $test_data (@$fixture) {
            my ($test_name, $action, $rule_id, $args, $expect_multistate, $expect_blocks) =
              @$test_data{qw(test_name  action  rule_id  args  expect_multistate  expect )};

            note "$counter. $test_name";
            subtest $test_name => sub {
                plan tests => 2;

                # Читаем текущий статус
                my $cur_multistate = 'New';
                if ($rule_id) {
                    $cur_multistate =
                      $app->business_rules->get($rule_id, fields => [qw(multistate_name)])->{'multistate_name'};
                }

                # Скидываем multistate
                $app->business_blocks->partner_db_table()
                  ->edit($app->partner_db->filter(['block_id', 'IS NOT', \undef]), {multistate => 0});

                # Редактируем
                if ($action eq 'add') {
                    $rule_id = $app->business_rules->add(%$args);
                } else {
                    $app->business_rules->do_action($rule_id, $action, %$args);
                }

                # Читаем новый статус
                my $got_multistate =
                  $app->business_rules->get($rule_id, fields => [qw(multistate_name)])->{'multistate_name'};

                $got_multistate =~ s/\n/ /g;

                is($got_multistate, $expect_multistate,
                    sprintf('Multistate "%s" -> "%s"', $cur_multistate, $expect_multistate));

                my $got_blocks = $app->business_blocks->get_all(
                    fields   => [qw( rule_id  block_public_id   is_deleted )],
                    filter   => {multistate => 'need_update'},
                    order_by => [qw( rule_id  page_id  block_id)]
                );

                # Проверка проставления need_update всем блокам, в том числе в других правилах
                eq_or_diff($got_blocks, $expect_blocks, $test_name, {context => 2});
            };
            $counter++;
        }

    },
);
