#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::Most;
use Test::Deep;

use qbit;

my $MOCKED_CONTEXT_ADBLOCK_PARTNER = 1016;
my $MOCKED_YAN_PARTNER             = 1009;

my $fixture = {

    ############ Conditions

    ### Browsers
    'conditions.browsers' => {

        'all available - correct' => {
            data   => {'conditions' => '{"browsers": [2,3,4,5,6,7,28,55,56,86,88]}',},
            errors => {},
        },
        'undef - correct' => {
            data   => {'conditions' => '{"browsers": null }',},
            errors => {},
        },
        'not an array - fail' => {
            data   => {'conditions'          => '{"browsers": {"2":"XXX"} }',},
            errors => {'conditions.browsers' => ['Data must be ARRAY',],},
        },
        'has string - fail' => {
            data   => {'conditions'            => '{"browsers": [2,"XXX"]}',},
            errors => {'conditions.browsers.1' => ['Must be an unsigned integer']},
        },
        'has unknown id - fail' => {
            data   => {'conditions'          => '{"browsers": [ 2, 100500 ]}',},
            errors => {'conditions.browsers' => ['Extra values: 100500',],},
        },
        'has wrong value type - fail' => {
            data   => {'conditions'            => '{"browsers": [2, ["XXX"] ]}',},
            errors => {'conditions.browsers.1' => ['Must be an unsigned integer',],},
        },
    },

    ### Regions
    'conditions.regions' => {

        'any numbers - correct' => {
            data   => {'conditions' => '{"regions": [1, 100500]}',},
            errors => {},
        },
        'undef - correct' => {
            data   => {'conditions' => '{"regions": null }',},
            errors => {},
        },
        'not an array - fail' => {
            data   => {'conditions'         => '{"regions": {"1":"100500"} }',},
            errors => {'conditions.regions' => ['Data must be ARRAY',],},
        },
        'has string - fail' => {
            data   => {'conditions'           => '{"regions": [2,"XXX"]}',},
            errors => {'conditions.regions.1' => ['Must be an unsigned integer']},
        },
        'has wrong value type - fail' => {
            data   => {'conditions'           => '{"regions": [2, ["XXX"] ]}',},
            errors => {'conditions.regions.1' => ['Must be an unsigned integer',],},
        },
    },

    ### Systems
    'conditions.systems' => {

        'all available - correct' => {
            data   => {'conditions' => '{"systems": [2,3,4,15,16,33]}',},
            errors => {},
        },
        'undef - correct' => {
            data   => {'conditions' => '{"systems": null }',},
            errors => {},
        },
        'not an array - fail' => {
            data   => {'conditions'         => '{"systems": {"1":"100500"} }',},
            errors => {'conditions.systems' => ['Data must be ARRAY',],},
        },
        'has string - fail' => {
            data   => {'conditions'           => '{"systems": [2,"XXX"]}',},
            errors => {'conditions.systems.1' => ['Must be an unsigned integer']},
        },
        'has unknown id - fail' => {
            data   => {'conditions'         => '{"systems": [ 2, 100500 ]}',},
            errors => {'conditions.systems' => ['Extra values: 100500',],},
        },
        'has wrong value type - fail' => {
            data   => {'conditions'           => '{"systems": [2, {"foo":"XXX"} ]}',},
            errors => {'conditions.systems.1' => ['Must be an unsigned integer',],},
        },
    },

    ### Devices
    'conditions.devices' => {

        'all available - correct' => {
            data   => {'conditions' => '{"devices": ["desktop","tablet","mobile","smarttv"]}',},
            errors => {},
        },
        'undef - correct' => {
            data   => {'conditions' => '{"devices": null }',},
            errors => {},
        },
        'not an array - fail' => {
            data   => {'conditions'         => '{"systems": {"devices":"100500"} }',},
            errors => {'conditions.systems' => ['Data must be ARRAY',],},
        },
        'has string - fail' => {
            data   => {'conditions'         => '{"devices": [ "desktop", "XXX"]}',},
            errors => {'conditions.devices' => ['Extra values: XXX',],},
        },
        'has wrong value type - fail' => {
            data   => {'conditions'           => '{"devices": [ "desktop", ["tablet"] ] }',},
            errors => {'conditions.devices.1' => ['Data must be SCALAR',],},
        },
    },

    ### URLS
    'conditions.urls' => {

        'valid urls, case insensitive - correct' => {
            data   => {'conditions' => '{"urls": ["http://foo.com", "HTTPS://bar.net"]}',},
            errors => {},
        },
        'undef - correct' => {
            data   => {'conditions' => '{"urls": null }',},
            errors => {},
        },
        'not an array - fail' => {
            data   => {'conditions'      => '{"urls": {"http://foo.com":"100500"} }',},
            errors => {'conditions.urls' => ['Data must be ARRAY',],},
        },
        'no http:// - correct' => {
            data   => {'conditions' => '{"urls": ["ya.ru"]}',},
            errors => {},
        },
        'ftp::// - correct' => {
            data   => {'conditions' => '{"urls": ["ftp://ya.ru"]}',},
            errors => {},
        },
        'has wrong value type - fail' => {
            data   => {'conditions'        => '{"urls": ["http://foo.com", ["http://bar.net"] ]}',},
            errors => {'conditions.urls.1' => ['Data must be SCALAR',],},
        },
        'empty - fail' => {
            data   => {'conditions'      => '{"urls": ["", "foo.bar"]}',},
            errors => {'conditions.urls' => ['One of URL is empty',],},
        },
        'too long - fail' => {
            data   => {'conditions'      => sprintf('{"urls": ["%s"]}',       'X' x 257)},
            errors => {'conditions.urls' => [sprintf('Url: "%s" is too long', 'X' x 257)],},
        },
    },

    ### Query arguments
    'conditions.query_args' => {

        'valid query_args.puid - correct' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"query_args": { "puid1": ["foo", "bar"], "puid64": ["baz"] } }',
            },
            errors => {},
        },
        'valid query_args.adb_enabled - correct' => {
            data => {
                'owner_id'   => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' => '{"query_args": { "adb_enabled": [1] } }',
            },
            errors => {},
        },
        'undef - correct' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"query_args": null }',
            },
            errors => {},
        },
        'not an object - fail' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"query_args": ["puid1"] }',
            },
            errors => {'conditions.query_args' => ['Data must be HASH',],},
        },
        'unknown arg - fail' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"query_args": { "puid99": ["foo"] } }',
            },
            errors => {'conditions.query_args' => ['Extra values: puid99',],},
        },
        'value not an array  - fail' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"query_args": { "puid1": "foo" } }',
            },
            errors => {'conditions.query_args' => ['Some parameter values are not arrays: "puid1"',],},
        },
        'has wrong value type - fail' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"query_args": { "puid1": ["foo"], "puid64": {"bar":"baz"} } }',
            },
            errors => {'conditions.query_args' => ['Some parameter values are not arrays: "puid64"',],},
        },
        'adb_enabled without "adblock_view" right - fail' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"query_args": { "adb_enabled": [1] } }',
            },
            errors => {'conditions.query_args' => ['Extra values: adb_enabled']},
        },
        'adb_enabled - has wong value [2]' => {
            data => {
                'owner_id'   => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' => '{"query_args": { "adb_enabled": [2] } }',
            },
            errors => {'conditions.query_args' => ['"adb_enabled" can not be set to a value over ["1"]: ["2"]',],},
        },
        'adb_enabled - has wong value [1,2]' => {
            data => {
                'owner_id'   => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' => '{"query_args": { "adb_enabled": [1,2] } }',
            },
            errors => {'conditions.query_args' => ['"adb_enabled" is not an array of only one value: ["1", "2"]',],},
        },
    },

    ### Headers
    'conditions.headers' => {
        'valid headers - correct' => {
            data => {
                'owner_id'   => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' => '{"headers": {"content-type": ["text/plain"], "accept": ["application/json"] } }',
            },
            errors => {},
        },
        'undef - correct' => {
            data => {
                'owner_id'   => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' => '{"headers": null }',
            },
            errors => {},
        },
        'not an object - fail' => {
            data => {
                'owner_id'   => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' => '{"headers": ["content-type", "text/plain"] }',
            },
            errors => {'conditions.headers' => ['Data must be HASH',],},
        },
        'value not an array  - fail' => {
            data => {
                'owner_id'   => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' => '{"headers": { "content-type" : "text/plain" } }',
            },
            errors => {'conditions.headers' => ['Not all values are arrays: "text/plain"',],},
        },
        'has wrong value type - fail' => {
            data => {
                'owner_id' => $MOCKED_CONTEXT_ADBLOCK_PARTNER,
                'conditions' =>
                  '{"headers": {"content-type": ["text/plain"], "accept": {"application/json":100500} } }',
            },
            errors => {'conditions.headers' => ['Not all values are arrays: {"application/json":100500}',],},
        },
        'without "adblock_view" right - fail' => {
            data => {
                'owner_id'   => $MOCKED_YAN_PARTNER,
                'conditions' => '{"headers": {"content-type": ["text/plain"], "accept": ["application/json"] } }',
            },
            errors => {'conditions.headers' => ['Access denied',],},
        },
    },
    'conditions.corner_cases' => {
        'empty json hash conditions' => {
            data   => {'conditions' => '{}'},
            errors => {},
        },
        'empty string conditions' => {
            data   => {'conditions' => ''},
            errors => {'conditions' => ['Data must be HASH']},
        },
        'undef conditions' => {
            data   => {'conditions' => undef},
            errors => {},
        },
        'zero cpm #PI-11990' => {
            data   => {'cpm' => 0, 'conditions' => '{}'},
            errors => {'cpm' => ["cpm value must be from '1' to '9999'"]},
        }
    }
};

run_tests(
    sub {
        my ($app) = @_;

        no strict 'refs';
        no warnings 'redefine';
        *{'QBit::Application::Model::RBAC::DB::get_rights_by_user_id'} = sub {
            my ($self, $user_id, $right_names) = @_;
            return $user_id == $MOCKED_CONTEXT_ADBLOCK_PARTNER
              ? {
                'context_on_site_adblock_view'                    => {},
                'business_rules_nobody-has-this-cool-feature-yet' => {}
              }
              : {};
        };

        foreach my $test_name (sort keys %$fixture) {

            subtest $test_name => sub {

                foreach my $sub_test_name (sort keys %{$fixture->{$test_name}}) {
                    my ($errors, $data) = @{$fixture->{$test_name}->{$sub_test_name}}{qw( errors data  )};
                    $data->{conditions} = from_json($data->{conditions}) if $data->{conditions};

                    _check_validation($app, $data, $errors, $test_name . '.' . $sub_test_name);
                }
              }
        }

    },
);

sub _check_validation {
    my ($app, $data, $expect_errors, $test_name) = @_;

    my $qv = QBit::Validator->new(
        data => $data,
        app  => $app->business_rules,
        $app->business_rules->get_template(fields => [keys %$data]),
    );

    my @errors     = $qv->get_fields_with_error();
    my $got_errors = _get_erros(\@errors);

    if (%$expect_errors) {
        my $is_ok = cmp_deeply($got_errors, $expect_errors, $test_name);
        warn Data::Dumper->Dump([$got_errors], ['$got_errors ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;
    } else {
        my $is_ok = is($qv->has_errors, FALSE, $test_name);
        warn Data::Dumper->Dump([$got_errors], ['$got_errors ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;
    }

    return 1;
}

sub _get_erros {
    my ($errors) = @_;

    my $res = {
        # <path> => [ 'msg1', ... ]
    };
    foreach my $row (@$errors) {
        my $key = join '.', @{$row->{path}};
        my $ar = $res->{$key} //= [];

        # ARRAY(0xf175be0) -> ARRAY(0xFFFFFF)
        push @$ar, map {s/\(0x[^)]+\)/(0xFFFFFF)/g; $_} @{$row->{msgs}};
    }

    return $res;
}
