#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Partner2::Simple;

use Test::More;
use Test::Exception;
use Test::MockObject::Extends;
use Test::Partner::Utils qw(get_new_user set_test_partner_utils_app );

my $uniq = 0;

run_tests(
    sub {
        my ($app) = @_;

        ### SETUP
        mock_app($app);
        $app->set_app_locale('en');

        # Excisting partner
        run_tests_for_logins($app, 'mocked-yan-manager', 'mocked-video-partner');

        # New partner
        my $new_user_login = _get_new_user_login($app);
        run_tests_for_logins($app, 'mocked-yan-manager', $new_user_login);

    },
    application_package => 'Rosetta',
    init                => [qw(api_balance)],
    mock_time           => 0
);

sub run_tests_for_logins {
    my ($app, $creator_login, $partner_login) = @_;

    subtest sprintf('Check inviter for manager "%s" and partner "%s"', $creator_login, $partner_login) => sub {

        # 1. МЕНЕДЖЕР создает invite - отправляется письмо на email партнера
        my $invitation_id = _do_check(\&check_add, $app, $creator_login);

        # 2. Партнер заходит по секретной ссылке из письма /v2/accept/inviter?id=...
        _do_check(\&check_link_is_valid, $app, $partner_login, $invitation_id);

        # 3. Партнер нажимает на кнопку "продолжить под этим логином"
        # 4. Фронт дергает accept
        _do_check(\&check_accept, $app, $partner_login, $invitation_id);

        # 5. Партнер заполняет миниформу, нажимает сохранить
        _do_check(\&check_save, $app, $partner_login, $invitation_id);
      }
}

sub check_add {
    my ($app, $user_login) = @_;

    my $invitation_id = $app->inviter->add(
        type    => 1,
        lang    => 'ru',
        email   => 'r' . $uniq++ . '@r.r',
        subject => 'Приглашение в партнерскую программу РСЯ',
        text =>
"Здравствуйте!\n\nДобро пожаловать Партнерскую программу Рекламной сети Яндекса. Для начала работы пройдите по ссылке чтобы начать регистрацию: :\n\n[% invite_link %]"
    );

    like($invitation_id, qr/^[0-9a-z]{32}$/, 'Invitation ID is a 32 length string');
    my $invite = $app->partner_db->invites->get($invitation_id, fields => ['lang']);
    is($invite->{lang}, 'ru', 'Invite lang is correct');

    return $invitation_id;
}

sub check_link_is_valid {
    my ($app, $user_login, $invitation_id) = @_;

    my $check_fail = $app->check_rights('yndx_login') ? 1 : 0;

    subtest 'Check "check_link_is_valid" method' => sub {

        my $res = eval {$app->inviter->check_link_is_valid($invitation_id)};

        if ($check_fail) {
            is(ref($@), 'Exception::Denied', 'Invite ID is denied for "YNDX_login"');
            is($@->message(), 'You do not have access to this page', 'Got proper error message');
        } else {
            is(ref($@), '', 'No exceptions');
            is($res,    1,  'Invitation ID is valid for PARTNER user');
        }
    };

}

sub check_accept {
    my ($app, $user_login, $invitation_id) = @_;

    subtest 'Check "accept" method' => sub {

        # 1. Check Invite ID is free
        {
            my $invite = _get_invite($app, $invitation_id, 'partner_id');

            is($invite->{partner_id}, undef, 'Invite ID is not binds to any user');
        }

        # 2. Do binding
        {
            my $cur_user = $app->get_option('cur_user');

            no strict 'refs';
            no warnings 'redefine';
            *{'QBit::Application::Model::API::Yandex::Balance::get_client_id_by_uid'} = sub {
                return $cur_user->{'client_id'};
            };

            *{'Application::get_user_lang'} = sub {
                return $app->get_option('locale');
            };

            my $res = $app->inviter->accept($invitation_id);
            is($res, 1, 'Accept is succesfull');
        }

        # 3.  Check Invite ID is bind to current user
        {
            my $invite = _get_invite($app, $invitation_id, 'partner_id');

            my $cur_user = $app->get_option('cur_user');

            is($invite->{partner_id}, $cur_user->{id}, 'Invite ID binds to current user successfully');
        }
    };
}

sub check_save {
    my ($app, $user_login, $invitation_id) = @_;

    subtest 'Check "save" method' => sub {

        my $invite_role_id = _get_invite_role_id($app, $invitation_id);

        # 1. Check user hasn't a role yet ($app->inviter->has_user_role_already)
        my $roles = $app->rbac->get_cur_user_roles();
        ok(!exists $roles->{$invite_role_id}, 'User has not a role yet');

        # 2. do "save"
        {
            my $cur_user = $app->get_option('cur_user');

            my %opts = (
                email      => $cur_user->{email},
                name       => $cur_user->{name},
                midname    => 'foo',
                lastname   => $cur_user->{lastname},
                phone      => '123213',
                newsletter => '0'
            );

            my $res = $app->inviter->save($invitation_id, %opts);

            is_deeply($res, {}, 'User registered successfully');
        }

        # 3. Check user got a role
        $roles = $app->rbac->get_cur_user_roles();
        ok(exists $roles->{$invite_role_id}, 'User got proper role');
    };

}

sub _do_check {
    my ($sub_ref, $app, $user_login, @args) = @_;

    # set new user if it changed
    change_cur_user($app, $user_login);

    ####### Check
    my $res = $sub_ref->($app, $user_login, @args);

    return $res;
}

sub _get_invite {
    my ($app, $invitation_id, @fields) = @_;

    my $tmp_rights = $app->add_tmp_rights(qw(inviter_view inviter_view_all));
    my $invite = $app->inviter->get($invitation_id, fields => \@fields);

    return $invite;
}

sub _get_invite_role_id {
    my ($app, $invitation_id) = @_;

    my $invite = _get_invite($app, $invitation_id, 'type');

    my $type_data      = $app->inviter->_get_type_by_id($invite->{type});
    my $invite_role_id = $type_data->{role_id};

    return $invite_role_id;
}

sub _get_new_user_login {
    my ($app) = @_;

    my $tmp_rights = $app->add_all_tmp_rights();

    set_test_partner_utils_app($app);

    my $cur_user = get_new_user(cur_user => 1);

    return $cur_user->{login};
}

sub mock_app {
    my ($app) = @_;

    $app->mailer;
    $app->{'mailer'} = Test::MockObject::Extends->new($app->{'mailer'});
    $app->mailer->mock('send', sub {1});
}
