#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Differences;
use Test::Exception;
use Test::More;

use Test::Partner2::Mock qw(mock_subs restore_subs);
use Test::Partner2::Simple;
use qbit;

my $MODEL                      = 'mobile_app_settings';
my @PAGE_IDS                   = sort qw(43569 154198 153979 153980 213953);
my $MOCKED_NORMAL_EDIT_PAGE    = [map {{PageID => $_, OptionsProtected => JSON::XS::false, ReadOnly => 0}} @PAGE_IDS];
my $MOCKED_PROTECTED_EDIT_PAGE = [map {{PageID => $_, OptionsProtected => JSON::XS::true, ReadOnly => 0}} @PAGE_IDS];
my $PAGE_ID                    = '43569';
my $MOCKED_SINGLE_EDIT_PAGE    = [
    {
        Options => '',
        PageID  => $PAGE_ID,
    },
];
my %EXPORT_TABLE_OPTS = (
    'field_names' => ['PageID', 'ReadOnly', 'OptionsProtected'],
    'table_name'  => 'Page'
);

run_tests(
    sub {
        my ($app) = @_;

        subtest default => sub {
            plan tests => 5;

            mock_subs(
                {
                    'QBit::Application::Model::API::Yandex::YT::BK::export_table' => sub {
                        my ($self, %opts) = @_;

                        eq_or_diff(\%opts, {%EXPORT_TABLE_OPTS, page_id => \@PAGE_IDS,}, 'export_table opts');

                        return $MOCKED_NORMAL_EDIT_PAGE;
                    },
                    'Application::Model::Page::ERROR' => sub {
                        fail 'error call';
                    },
                }
            );
            my ($checked, $fixed, $failed) = $app->$MODEL->check_read_only_and_protected_status();
            is($checked, 5,     'checked pages');
            is($fixed,   0,     'fixed pages');
            is($failed,  FALSE, 'failed');
            is(
                scalar(
                    grep {!$app->$MODEL->check_multistate_flag($_->{multistate}, 'protected')} @{
                        $app->$MODEL->get_all(
                            fields => ['multistate'],
                            filter => [$app->$MODEL->get_page_id_field_name() => 'IN' => \@PAGE_IDS]
                        )
                      }
                ),
                scalar @PAGE_IDS,
                'state is not changed'
              );
            restore_subs(
                ['QBit::Application::Model::API::Yandex::YT::BK::export_table', 'Application::Model::Page::ERROR',]);
        };

        subtest page_id => sub {
            plan tests => 5;

            mock_subs(
                {
                    'QBit::Application::Model::API::Yandex::YT::BK::export_table' => sub {
                        my ($self, %opts) = @_;

                        eq_or_diff(\%opts, {%EXPORT_TABLE_OPTS, page_id => [$PAGE_ID],}, 'export_table opts');

                        return $MOCKED_SINGLE_EDIT_PAGE;
                    },
                    'Application::Model::Page::ERROR' => sub {
                        fail 'error call';
                    },
                }
            );
            my ($checked, $fixed, $failed) = $app->$MODEL->check_read_only_and_protected_status(page_id => [$PAGE_ID]);
            is($checked, 1,     'checked pages');
            is($fixed,   0,     'fixed pages');
            is($failed,  FALSE, 'failed');

            is(
                scalar(
                    grep {!$app->$MODEL->check_multistate_flag($_->{multistate}, 'protected')} @{
                        $app->$MODEL->get_all(
                            fields => ['multistate'],
                            filter => {$app->$MODEL->get_page_id_field_name() => $PAGE_ID}
                        )
                      }
                ),
                1,
                'state is not changed'
              );
            restore_subs(
                ['QBit::Application::Model::API::Yandex::YT::BK::export_table', 'Application::Model::Page::ERROR',]);
        };

        subtest page_ids => sub {
            plan tests => 5;

            mock_subs(
                {
                    'QBit::Application::Model::API::Yandex::YT::BK::export_table' => sub {
                        my ($self, %opts) = @_;

                        eq_or_diff(\%opts, {%EXPORT_TABLE_OPTS, page_id => \@PAGE_IDS,}, 'export_table opts');

                        return $MOCKED_NORMAL_EDIT_PAGE;
                    },
                    'Application::Model::Page::ERROR' => sub {
                        fail 'error call';
                    },
                }
            );
            my ($checked, $fixed, $failed) = $app->$MODEL->check_read_only_and_protected_status(page_id => \@PAGE_IDS);
            is($checked, 5,     'checked pages');
            is($fixed,   0,     'fixed pages');
            is($failed,  FALSE, 'failed');

            is(
                scalar(
                    grep {!$app->$MODEL->check_multistate_flag($_->{multistate}, 'protected')} @{
                        $app->$MODEL->get_all(
                            fields => ['multistate'],
                            filter => [$app->$MODEL->get_page_id_field_name() => 'IN' => \@PAGE_IDS]
                        )
                      }
                ),
                scalar @PAGE_IDS,
                'state is not changed'
              );
            restore_subs(
                ['QBit::Application::Model::API::Yandex::YT::BK::export_table', 'Application::Model::Page::ERROR',]);
        };

        subtest fixed => sub {
            plan tests => 5;

            mock_subs(
                {
                    'QBit::Application::Model::API::Yandex::YT::BK::export_table' => sub {
                        my ($self, %opts) = @_;

                        eq_or_diff(\%opts, {%EXPORT_TABLE_OPTS, page_id => \@PAGE_IDS,}, 'export_table opts');

                        return $MOCKED_PROTECTED_EDIT_PAGE;
                    },
                    'Application::Model::Page::ERROR' => sub {
                        fail 'error call';
                    },
                }
            );
            my ($checked, $fixed, $failed) = $app->$MODEL->check_read_only_and_protected_status();
            is($checked, 5,     'checked pages');
            is($fixed,   5,     'fixed pages');
            is($failed,  FALSE, 'failed');

            is(
                scalar(
                    grep {$app->$MODEL->check_multistate_flag($_->{multistate}, 'protected')} @{
                        $app->$MODEL->get_all(
                            fields => ['multistate'],
                            filter => [$app->$MODEL->get_page_id_field_name() => 'IN' => \@PAGE_IDS]
                        )
                      }
                ),
                scalar @PAGE_IDS,
                'state is not changed'
              );
            restore_subs(
                ['QBit::Application::Model::API::Yandex::YT::BK::export_table', 'Application::Model::Page::ERROR',]);
        };

        subtest 'fatal exception' => sub {
            plan tests => 5;

            mock_subs(
                {
                    'QBit::Application::Model::API::Yandex::YT::select_from_any_replica' => sub {
                        return 'invalid json';
                    },
                    'Application::Model::PageMinimal::ERROR' => sub {
                        my ($exception) = @_;

                        isa_ok($exception->{exception}, 'Exception::API::YT::BK');
                        is($exception->{message},
                            'check_read_only_and_protected_status FAILED for "mobile_app_settings"');
                    },
                }
            );
            my ($checked, $fixed, $failed) = $app->$MODEL->check_read_only_and_protected_status();
            is($checked, 5,    'checked pages');
            is($fixed,   0,    'fixed pages');
            is($failed,  TRUE, 'failed');
            restore_subs(
                [
                    'QBit::Application::Model::API::Yandex::YT::select_from_any_replica',
                    'Application::Model::PageMinimal::ERROR',
                ]
            );
        };

        subtest 'exception with result' => sub {
            plan tests => 4;

            local $MOCKED_NORMAL_EDIT_PAGE->[-1]{PageID} = '123';
            mock_subs(
                {
                    'QBit::Application::Model::API::Yandex::YT::BK::export_table' => sub {

                        return $MOCKED_NORMAL_EDIT_PAGE;
                    },
                    'Application::Model::PageMinimal::ERROR' => sub {
                        my ($exception) = @_;

                        eq_or_diff(
                            $exception,
                            {
                                message => $MODEL . ': requested and received page_ids are different',
                                extra   => {
                                    extra_pages =>
                                      {123 => {PageID => '123', OptionsProtected => JSON::XS::false, ReadOnly => 0}},
                                    ignored_pages => [$PAGE_ID],
                                },
                            },
                            'Correct data of error'
                        );
                    },
                }
            );
            my ($checked, $fixed, $failed) = $app->$MODEL->check_read_only_and_protected_status();
            is($checked, 5,    'checked pages');
            is($fixed,   4,    'fixed pages');
            is($failed,  TRUE, 'failed');
            restore_subs(
                ['QBit::Application::Model::API::Yandex::YT::BK::export_table', 'Application::Model::Page::ERROR',]);
        };

    },
    init => [qw(api_yt_bk mobile_app_settings api_yt)],
);
