use strict;
use warnings FATAL => 'all';

use qbit;
use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_bk mock_balance mock_utils_partner2 mock_format_system mock_yt_picategory_mapping);
use Test::Partner::Utils qw(get_new_an_rtb);

use Test::More;
use Test::Deep qw(cmp_deeply);

use PiConstants qw($TECHNICAL_RTB_BLOCK_ID);

my $VIP_CAMPAIGN_ID;
my $BUSINESS_UNIT_CAMPAIGN_ID;
my $PAGE_ID = 41443;
my $DSP_ID  = 2317563;

run_tests(
    sub {
        my ($app) = @_;

        mock_bk($app);
        mock_balance($app);
        mock_utils_partner2($app);
        mock_format_system($app);
        mock_yt_picategory_mapping($app);

        add_context_pages($app);

        add_rtb_block_for_vip_page($app);

        add_rtb_block_for_user_with_business_unit_flag($app);

        add_technical_rtb_block($app);

        add_not_active_rtb_block($app);

        check_rtb_blocks_before($app);

        change_cur_user($app, 'mocked-dsp-manager');
        my $tmp_rights = $app->add_tmp_rights(qw(context_on_site_campaign_view_all users_view_all));

        $app->context_on_site_rtb->turn_on_dsp(
            {
                id         => $DSP_ID,
                types      => [0],
                multistate => $app->dsp->get_multistates_by_filter(
                    'created_in_bk and (created_in_pi or not_need_create_in_pi) and linked_in_balance and not deleted')
                  ->[0],
                tag => 'test'
            }
        );

        check_after_turn_on($app);
    },
);

sub add_context_pages {
    my ($app) = @_;

    $VIP_CAMPAIGN_ID =
      $app->context_on_site_campaign->add(domain_id => 3, caption => 'VIP', login => 'tutby-partner-with-agreement');
    $app->partner_db->vip->add(
        {page_id => $app->context_on_site_campaign->get($VIP_CAMPAIGN_ID, fields => [qw(page_id)])->{'page_id'}});

    $BUSINESS_UNIT_CAMPAIGN_ID = $app->context_on_site_campaign->add(
        domain_id => 1,
        caption   => 'BUSINESS UNIT',
        login     => 'mocked-context-adblock-partner'
    );
    $app->partner_db->users->edit($app->partner_db->filter({login => 'mocked-context-adblock-partner'}),
        {business_unit => 1});

    $app->context_on_site_campaign->cron_init_campaigns();
}

sub add_rtb_block_for_vip_page {
    my ($app) = @_;

    my $page_id = $app->context_on_site_campaign->get($VIP_CAMPAIGN_ID, fields => [qw(page_id)])->{'page_id'};

    $app->context_on_site_rtb->add(%{get_new_an_rtb({campaign_id => $page_id, 'dsp_mode' => 'auto'})});
}

sub add_rtb_block_for_user_with_business_unit_flag {
    my ($app) = @_;

    my $page_id = $app->context_on_site_campaign->get($BUSINESS_UNIT_CAMPAIGN_ID, fields => [qw(page_id)])->{'page_id'};

    $app->context_on_site_rtb->add(%{get_new_an_rtb({campaign_id => $page_id})});
}

sub add_technical_rtb_block {
    my ($app) = @_;

    my $rtb = {
        hash_transform(
            get_new_an_rtb({campaign_id => $PAGE_ID}),
            [map {$_->name} @{$app->partner_db->context_on_site_rtb->fields}]
        )
    };

    $rtb->{'id'} = $TECHNICAL_RTB_BLOCK_ID;

    $app->partner_db->context_on_site_rtb->add($rtb);
}

sub add_not_active_rtb_block {
    my ($app) = @_;

    my $public_id = $app->context_on_site_rtb->add(%{get_new_an_rtb({campaign_id => $PAGE_ID})});

    my $id = $app->context_on_site_rtb->_split_id($public_id);

    $app->partner_db->context_on_site_rtb->edit($id, {active => 0});
}

sub check_rtb_blocks_before {
    my ($app) = @_;

    my $dsps = $app->partner_db->block_dsps->get_all(
        filter   => {dsp_id => $DSP_ID},
        order_by => [qw(page_id block_id)]
    );

    cmp_deeply(
        $dsps,
        [
            {
                'block_id'   => '42',
                'dsp_id'     => '2317563',
                'interval'   => undef,
                'is_deleted' => '0',
                'page_id'    => '41443',
                'show_count' => undef
            }
        ],
        'check rtb blocks before'
    );
}

sub check_after_turn_on {
    my ($app) = @_;

    my $dsps = $app->partner_db->block_dsps->get_all(
        fields   => [qw(page_id block_id is_deleted)],
        filter   => {dsp_id => $DSP_ID},
        order_by => [qw(page_id block_id)]
    );

    cmp_deeply(
        $dsps,
        [
            {
                'block_id'   => '1',
                'is_deleted' => '0',
                'page_id'    => '41443'
            },
            {
                'block_id'   => '42',
                'is_deleted' => '0',
                'page_id'    => '41443'
            },
            {
                'block_id'   => '100000',
                'is_deleted' => '0',
                'page_id'    => '41443'
            },
            {
                'block_id'   => '1',
                'is_deleted' => '0',
                'page_id'    => '142898'
            }
        ],
        'check rtb blocks after'
    );

    my $pages = $app->partner_db->need_update_pages->get_all(
        fields   => [qw(page_id)],
        filter   => {processed => 0},
        order_by => [qw(page_id)]
    );

    cmp_deeply($pages, [{'page_id' => '41443'}, {'page_id' => '142898'}], 'check pages after');
}
