use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Mock;
use Test::Partner::Utils qw(get_new_internal_direct_context);
use qbit qw(from_json);

use Test::More;
use Test::Deep qw(cmp_deeply);

our $PAGE_ID = 148805;
our $PUBLIC_ID;

sub clear_available_editable_fields {
    my ($block) = @_;

    my @fields = qw(bk_data is_custom_bk_data);

    foreach my $k (keys(%{$block->{available_fields}})) {
        if (!grep {$k eq $_} @fields) {
            delete($block->{available_fields}->{$k});
        }
    }

    foreach my $k (keys(%{$block->{editable_fields}})) {
        if (!grep {$k eq $_} @fields) {
            delete($block->{editable_fields}->{$k});
        }
    }

    return 1;
}

sub create_new_block {
    my ($app) = @_;

    $PUBLIC_ID = $app->internal_context_on_site_direct->add(
        get_new_internal_direct_context(
            {
                campaign_id          => $PAGE_ID,
                is_internal          => 1,
                type                 => 'vertical',
                url_background_color => '0000CC',
            }
        )
    );

    return 1;
}

sub check_initial_block_data {
    my ($app) = @_;

    my $initial_block_data =
      $app->internal_context_on_site_direct->get($PUBLIC_ID,
        fields => [qw(is_custom_bk_data bk_data available_fields editable_fields)]);
    clear_available_editable_fields($initial_block_data);

    cmp_deeply(
        $initial_block_data,
        {
            is_custom_bk_data => 0,
            bk_data           => undef,
            available_fields  => {
                bk_data           => 1,
                is_custom_bk_data => 1,
            },
            editable_fields => {is_custom_bk_data => 1,},
        },
        'expected block data after block creation',
    );

    return 1;
}

sub send_page_to_bk {
    my ($app) = @_;

    $app->internal_context_on_site_campaign($PAGE_ID, 'set_need_update');
    $app->internal_context_on_site_campaign->update_in_bk({id => $PAGE_ID});

    return 1;
}

sub check_block_data_after_bk_send {
    my ($app) = @_;

    my $block_data_after_bk_send =
      $app->internal_context_on_site_direct->get($PUBLIC_ID,
        fields => [qw(is_custom_bk_data bk_data available_fields editable_fields)]);
    clear_available_editable_fields($block_data_after_bk_send);

    # разворачиваю json данные в perl структуру и удаляю почти все поля (чтобы реже нужно было этот тест править)
    $block_data_after_bk_send->{bk_data} = from_json($block_data_after_bk_send->{bk_data});
    my $design = $block_data_after_bk_send->{bk_data}->{Design}{0}{design};
    foreach my $k (keys(%$design)) {
        delete($design->{$k}) if $k ne 'blockId';
    }

    cmp_deeply(
        $block_data_after_bk_send,
        {
            bk_data => {
                BlockModel => 'internal_context_on_site_direct',
                Design     => {0 => {name => "", design => {blockId => $PUBLIC_ID,}}},
            },
            is_custom_bk_data => 0,
            available_fields  => {
                bk_data           => 1,
                is_custom_bk_data => 1,
            },
            editable_fields => {is_custom_bk_data => 1,},
        },
        'expected block data after bk send',
    );

    return 1;
}

sub edit_bk_data {
    my ($app) = @_;

    $app->internal_context_on_site_direct->do_action(
        $PUBLIC_ID,
        'edit',
        is_custom_bk_data => 1,
        bk_data           => '{"Design":{"0":{"name":"","design":{}}},"secret_option":12}',
    );

    return 1;
}

sub check_block_data_after_edit {
    my ($app) = @_;

    my $block_data_after_god_mode =
      $app->internal_context_on_site_direct->get($PUBLIC_ID,
        fields => [qw(is_custom_bk_data bk_data available_fields editable_fields)]);
    clear_available_editable_fields($block_data_after_god_mode);

    cmp_deeply(
        $block_data_after_god_mode,
        {
            bk_data => '{
   "Design" : {
      "0" : {
         "design" : {},
         "name" : ""
      }
   },
   "secret_option" : 12
}
',
            is_custom_bk_data => 1,
            available_fields  => {
                bk_data           => 1,
                is_custom_bk_data => 1,
            },
            editable_fields => {
                bk_data           => 1,
                is_custom_bk_data => 1,
            },
        },
        'expected block data after setting god mode',
    );

    return 1;
}

sub add_manager {
    my ($app) = @_;

    $app->partner_db->managers->add({manager_id => 7169516, page_id => $PAGE_ID});

    return 1;
}

sub check_block_data_under_manager {
    my ($app) = @_;

    my $block_data =
      $app->internal_context_on_site_direct->get($PUBLIC_ID,
        fields => [qw(is_custom_bk_data bk_data available_fields editable_fields actions)]);
    clear_available_editable_fields($block_data);

    my $actions = delete($block_data->{actions});

    # PI-23713
    ok(exists($actions->{edit}), 'manager should have action "edit"');

    cmp_deeply(
        $block_data,
        {
            bk_data => '{
   "Design" : {
      "0" : {
         "design" : {},
         "name" : ""
      }
   },
   "secret_option" : 12
}
',
            is_custom_bk_data => 1,
            available_fields  => {
                is_custom_bk_data => 1,
                bk_data           => 1
            },
            editable_fields => {
                is_custom_bk_data => 1,
                bk_data           => 1
            },
        },
        'expected block data under manager',
    );

    return 1;
}

run_tests(
    sub {
        my ($app) = @_;

        mock_bk($app);

        $app->{'__CURRENT_USER_RIGHTS__'} = {};

        my $yandex_service_administartor_user_id = 1019;
        $app->set_cur_user({id => $yandex_service_administartor_user_id});

        create_new_block($app);
        check_initial_block_data($app);
        send_page_to_bk($app);
        check_block_data_after_bk_send($app);
        edit_bk_data($app);
        check_block_data_after_edit($app);

        add_manager($app);

        my $yandex_service_manager_user_id = 7169516;
        $app->set_cur_user({id => $yandex_service_manager_user_id});

        check_block_data_under_manager($app);

    },
);
