#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::Most;
use Test::Deep;

use qbit;

my $media_cpm_error       = "media_cpm value must be from '0.001' to '9999'";
my $text_cpm_error        = "text_cpm value must be from '0.001' to '9999'";
my $mincpm_error          = "mincpm value must be from '0' to '9999'";
my $percent_traffic_error = "Percent traffic value must be from '1' to '99'";
my $articles_error        = qq(Not match with schema
[
   {
      "instance" : %s,
      "keyword" : "%s",
      "message" : "%s",
      "pointer" : "/0/cpm",
      "schema" : {
         "type" : "integer"
      },
      "schema_pointer" : "/items/properties/cpm",
      "schema_pointer_history" : []
   }
]
);

my $fixture = {

    #Cases

    #dsps
    'cases.dsps' => {
        'dsps not exists' => {
            data   => {'dsps' => ["25630811"], 'show_video' => 0,},
            errors => {'dsps' => ["DSP's with ID 25630811 not found"]},
        },
    },

    #geo
    'cases.geo' => {
        'geo not exist' => {
            data   => {'geo' => to_json([{cpm => 3, id => "3000000"}])},
            errors => {'geo' => ["Entry with id: '3000000' not found"]},
        },
    },

    #god mod
    'cases.god_mod' => {
        'god mod incorrect' => {
            data   => {'bk_data' => "", 'is_custom_bk_data' => 1},
            errors => {'bk_data' => ["Incorrect json"]},
        },
    },

    #media_cpm
    'cases.media_cpm' => {
        'zero media_cpm' => {
            data   => {'media_cpm' => 0},
            errors => {'media_cpm' => [$media_cpm_error]},
        },
        'null media_cpm' => {
            data   => {'media_cpm' => "null"},
            errors => {'media_cpm' => [$media_cpm_error]},
        },
        'negative media_cpm' => {
            data   => {'media_cpm' => -1},
            errors => {'media_cpm' => [$media_cpm_error]},
        },
        'letter media_cpm' => {
            data   => {'media_cpm' => "a"},
            errors => {'media_cpm' => [$media_cpm_error]},
        },
        'too big media_cpm' => {
            data   => {'media_cpm' => "100000"},
            errors => {'media_cpm' => [$media_cpm_error]},
        },
    },

    #text_cpm
    'cases.text_cpm' => {
        'zero text_cpm' => {
            data   => {'text_cpm' => 0},
            errors => {'text_cpm' => [$text_cpm_error]},
        },
        'negative text_cpm' => {
            data   => {'text_cpm' => -1},
            errors => {'text_cpm' => [$text_cpm_error]},
        },
        'null text_cpm' => {
            data   => {'text_cpm' => "null"},
            errors => {'text_cpm' => [$text_cpm_error]},
        },
        'text text_cpm' => {
            data   => {'text_cpm' => "a"},
            errors => {'text_cpm' => [$text_cpm_error]},
        },
        'too big text_cpm' => {
            data   => {'text_cpm' => "100000"},
            errors => {'text_cpm' => [$text_cpm_error]},
        },
    },

    #mincpm
    'cases.mincpm' => {
        'negative mincpm' => {
            data   => {'mincpm' => -1},
            errors => {'mincpm' => [$mincpm_error]},
        },
        'too big mincpm' => {
            data   => {'mincpm' => "100000"},
            errors => {'mincpm' => [$mincpm_error]},
        },
        'null mincpm' => {
            data   => {'mincpm' => "null"},
            errors => {'mincpm' => [$mincpm_error]},
        },
        'letter mincpm' => {
            data   => {'mincpm' => "a"},
            errors => {'mincpm' => [$mincpm_error]},
        },
    },

    #brand
    'cases.brand' => {
        'brands not exists' => {
            data   => {'brands' => [{bid => "41954100", blocked => 0, cpm => 4}]},
            errors => {'brands' => ["Entry with id: '41954100' not found"]},
        },
    },
    #articles
    'cases.articles' => {
        'zero cpm' => {
            data => {'picategories' => [{cpm => 0, id => "41"}]},
            errors => {'picategories.0.cpm' => [q(Field 'cpm' should be a positive number less than 9999)],},
        },
        'letter cpm' => {
            data => {'picategories' => [{cpm => "a", id => "41"}]},
            errors => {'picategories.0.cpm' => [q[Field 'cpm' should be a positive number less than 9999]]},
        },
        'too big cpm' => {
            data => {'picategories' => [{cpm => 100_000, id => "41"}]},
            errors => {'picategories.0.cpm' => [q(Field 'cpm' should be a positive number less than 9999)],},
        },
        'null cpm' => {
            data => {'picategories' => [{cpm => "null", id => "41"}]},
            errors => {'picategories.0.cpm' => [q[Field 'cpm' should be a positive number less than 9999]]},
        },
        'negative cpm' => {
            data => {'picategories' => [{cpm => -1, id => "41"}]},
            errors => {'picategories.0.cpm' => [q(Field 'cpm' should be a positive number less than 9999)],},
        },
        'not exist id' => {
            data => {'picategories' => [{cpm => 555, id => "99999999410000999922223232"}]},
            errors => {'picategories' => ["Entry with id: '99999999410000999922223232' not found"]},
        },
    },
};

run_tests(
    sub {
        my ($app) = @_;

        foreach my $test_name (sort keys %$fixture) {

            subtest $test_name => sub {
                foreach my $sub_test_name (sort keys %{$fixture->{$test_name}}) {
                    my ($errors, $data) = @{$fixture->{$test_name}->{$sub_test_name}}{qw( errors data  )};

                    $data->{'site_version'} = 'desktop';
                    $data->{'page_id'}      = '88848';

                    _check_validation($app, $data, $errors, $test_name . '.' . $sub_test_name);
                }
              }
        }
    },
);

sub _check_validation {
    my ($app, $data, $expect_errors, $test_name) = @_;

    my $qv = $app->validator->check(
        data => $data,
        app  => $app->internal_context_on_site_rtb,
        $app->internal_context_on_site_rtb->get_template(fields => [keys %$data]),
    );

    my @errors     = $qv->get_fields_with_error();
    my $got_errors = _get_erros(\@errors);

    if (%$expect_errors) {
        my $is_ok = cmp_deeply($got_errors, $expect_errors, $test_name);
        warn Data::Dumper->Dump([$got_errors], ['$got_errors ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;
    } else {
        my $is_ok = is($qv->has_errors, FALSE, $test_name);
        warn Data::Dumper->Dump([$got_errors], ['$got_errors ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;
    }

    return 1;
}

sub _get_erros {
    my ($errors) = @_;

    my $res = {
        # <path> => [ 'msg1', ... ]
    };
    foreach my $row (@$errors) {
        my $key = join '.', @{$row->{path}};
        my $ar = $res->{$key} //= [];

        # ARRAY(0xf175be0) -> ARRAY(0xFFFFFF)
        push @$ar, map {s/\(0x[^)]+\)/(0xFFFFFF)/g; $_} @{$row->{msgs}};
    }

    return $res;
}
