#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Data::Rmap qw(rmap_hash);
use Test::Differences;
use Test::More;
use Test::Partner2::Mock;
use Test::Partner2::Simple;
use Test::Partner::Utils qw(get_test_data_and_update_if_needed);

use PiConstants qw($IS_PRECISE);

my @fields = qw(
  completed_percent
  error_data
  id
  log
  method_type
  method_name
  group_id
  multistate
  multistate_name
  params
  result
  user_id
  add_dt
  start_dt
  end_dt
  tries
  );

run_tests(
    sub {
        my ($app) = @_;

        my @got_data;
        mock_log(\@got_data);

        mock_curdate('2018-02-10 15:00:00');

        run_for_moderator(
            $app,
            sub {
                $app->queue->add(
                    method_name => 'update_mobile_app_status',
                    params      => {app_id => 1, active => 1},
                    group_id    => 1,
                );
                push @got_data, {add_task_1 => $app->queue->get_all(fields => \@fields)};
            }
        );

        run_for_cron(
            $app,
            sub {
                set_request_return_500();

                mock_curdate('2018-02-10 15:03:00');

                $app->queue->restart_lost_tasks();
                my $task = $app->queue->grab_next_task();
                push @got_data, {task_1_id => $task->{id}};
                $app->queue->start($task);
                push @got_data, {task_1_500 => $app->queue->get_all(fields => \@fields)};

                push @got_data, {no_task_1 => ($app->queue->grab_next_task() // {})->{id}};

                $app->queue->restart_lost_tasks();
                push @got_data, {no_task_1_again => ($app->queue->grab_next_task() // {})->{id}};

                mock_curdate('2018-02-10 15:05:00');

                set_request_return_ok();

                $app->queue->restart_lost_tasks();
                push @got_data, {task_1_restart => $app->queue->get_all(fields => \@fields)};

                $task = $app->queue->grab_next_task();
                push @got_data, {has_task_1 => $task->{id}};
                $app->queue->start($task);
                push @got_data, {task_1_done => $app->queue->get_all(fields => \@fields)};
            }
        );

        mock_curdate('2018-02-10 16:00:00');
        run_for_moderator(
            $app,
            sub {
                $app->queue->add(
                    method_name => 'update_mobile_app_status',
                    params      => {app_id => 1, active => 1},
                );
                push @got_data, {add_task_2 => $app->queue->get_all(fields => \@fields)};
            }
        );

        run_for_cron(
            $app,
            sub {
                set_request_die();

                for (1 .. 100) {
                    $app->queue->restart_lost_tasks();
                    my $task = $app->queue->grab_next_task();
                    $app->queue->start($task);
                    mock_curdate(date_add(curdate(), second => 90, oformat => 'db_time'));
                }
                push @got_data, {task_2_die => $app->queue->get_all(fields => \@fields)};
            }
        );

        my $expected = get_test_data_and_update_if_needed('data.json', \@got_data);
        if ($IS_PRECISE) {
            my $fix_data_sub = sub {
                my ($el) = @_;
                my @fields_to_num = qw(group_id id multistate tries user_id task_1_id has_task_1);
                map {$el->{$_} += 0}
                  grep {defined $el->{$_}} @fields_to_num;
                if (defined $el->{result}) {
                    $el->{result} =~ s/"(\d+)"/$1/g;
                }
                $el;
            };
            rmap_hash {$fix_data_sub->($_)} @got_data;
            rmap_hash {$fix_data_sub->($_)} @$expected;
        }
        eq_or_diff(from_json(to_json(\@got_data)), $expected, 'compare', {context => 10});

    },
);

sub run_for_moderator {
    my ($app, $sub) = @_;

    change_cur_user($app, 'mocked-moderator');

    $sub->();
}

sub run_for_cron {
    my ($app, $sub) = @_;

    change_cur_user($app, 'system-cron');
    no warnings 'redefine';
    local *QBit::Application::check_rights = sub {TRUE};

    $sub->();
}

sub set_request_return_ok {
    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my $r = HTTP::Response->new(200);
        $r->request(HTTP::Request->new());
        $r->content('{"data":{"AdfoxInternal":{"updateMobileAppStatus":true}}}');
        return $r;
    };
}

sub set_request_return_500 {
    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my $r = HTTP::Response->new(500);
        $r->request(HTTP::Request->new());
        return $r;
    };
}

sub set_request_die {
    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {die "forced die\n"};
}

sub mock_log {
    my ($data) = @_;
    no warnings 'redefine';
    no strict 'refs';
    require QBit::Application::Model::API::HTTP;
    *{'QBit::Application::Model::API::HTTP::INFO'} = sub { };
    require Application::Model::Queue;
    *{'Application::Model::Queue::ERROR'} = sub {
        push @$data, {ERROR => $_[0]->message};
    };
}
