#!/usr/bin/perl -w

# Usage: > ./prove2 *.t :: --self_update 2>&1 | less

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Utils qw(clean_for_json);
use Test::Partner2::Mock;
use Test::Partner::Utils qw(
  get_bk_data
  get_test_data_and_update_if_needed
  use_perl_do_actions
  );

use Test::Differences;
use Test::More tests => 5;

use qbit;

my $cpm_fields = [qw(mincpm video_cpm media_cpm text_cpm geo brands picategories)];
my $tests      = [
    {
        name       => 'context_on_site_rtb_mincpm',
        model      => 'context_on_site_rtb',
        cpm_fields => [qw(mincpm video_cpm media_cpm text_cpm geo brands picategories)],
        block_data => {
            blind            => 0,
            caption          => 'test',
            dsp_blocks       => ["728x90"],
            media_active     => undef,
            media_blocked    => undef,
            media_cpm        => undef,
            mincpm           => 10,
            page_id          => 153745,
            site_version     => "desktop",
            strategy         => 0,
            text_active      => undef,
            text_blocked     => undef,
            text_cpm         => undef,
            design_templates => [
                {
                    caption         => "Исходный дизайн",
                    design_settings => {
                        limit => 2,
                        name  => "240x400"
                    },
                    page_id                 => 153745,
                    is_custom_format_direct => 0,
                    type                    => "tga"
                },
                {
                    caption         => "Медиа дизайн",
                    design_settings => {
                        filterSizes     => JSON::XS::false,
                        horizontalAlign => JSON::XS::true,
                    },
                    page_id => 153745,
                    type    => "media",
                },
            ]
        },
        geo          => "[{\"cpm\":15,\"id\":\"10174\"},{\"cpm\":20,\"id\":\"1\"}]",
        picategories => [{"cpm" => 50, "id" => 1}, {"cpm" => 45, "id" => 2}],
        brands => [{bid => 420, blocked => 0, cpm => 120,},],
    },
    {
        name       => 'context_on_site_rtb_separate_cpm',
        model      => 'context_on_site_rtb',
        cpm_fields => [qw(mincpm video_cpm media_cpm text_cpm geo brands picategories)],
        block_data => {
            blind            => 0,
            caption          => 'test',
            dsp_blocks       => ["728x90"],
            media_active     => 1,
            media_blocked    => 0,
            media_cpm        => 33,
            page_id          => 153745,
            site_version     => "desktop",
            strategy         => 3,
            text_active      => 1,
            text_blocked     => 0,
            text_cpm         => 22,
            design_templates => [
                {
                    caption         => "Исходный дизайн",
                    design_settings => {
                        limit => 2,
                        name  => "240x400"
                    },
                    page_id                 => 153745,
                    is_custom_format_direct => 0,
                    type                    => "tga"
                },
                {
                    caption         => "Медиа дизайн",
                    design_settings => {
                        filterSizes     => JSON::XS::false,
                        horizontalAlign => JSON::XS::true,
                    },
                    page_id => 153745,
                    type    => "media",
                },
            ]
        },
        geo          => "[{\"cpm\":15,\"id\":\"10174\"},{\"cpm\":20,\"id\":\"1\"}]",
        picategories => [{"cpm" => 50, "id" => 1}, {"cpm" => 45, "id" => 2}],
        brands => [{bid => 420, blocked => 0, cpm => 120,},],
    },
    {
        name       => 'mobile_app_rtb_mincpm',
        model      => 'mobile_app_rtb',
        cpm_fields => [qw(mincpm media_cpm text_cpm geo brands picategories)],
        block_data => {
            block_type => "banner",
            caption    => "RTB block",
            page_id    => 43569,
            dsps       => [1],
            mincpm     => 11,
            show_video => 0,
            strategy   => 0,
        },
        geo          => "[{\"cpm\":15,\"id\":\"10174\"},{\"cpm\":20,\"id\":\"1\"}]",
        picategories => [{"cpm" => 50, "id" => 1}, {"cpm" => 45, "id" => 2}],
        brands => [{bid => 420, blocked => 0, cpm => 120,},],
    },
];

run_tests(
    sub {
        my ($app) = @_;

        mock_bk($app);
        mock_format_system($app);
        mock_utils_partner2($app);
        mock_yt_picategory_mapping($app);
        use_perl_do_actions($app, 'context_on_site_rtb');
        use_perl_do_actions($app, 'mobile_app_rtb');

        $app->set_cur_user({id => 0, login => 'system-cron'});

        for my $test (@$tests) {
            subtest $test->{name} => sub {
                my $model    = $test->{model};
                my $got_data = convert_block($app->$model, $test);
                my $has_data = get_test_data_and_update_if_needed("$test->{name}.json", $got_data);
                eq_or_diff(
                    from_json(to_json($got_data, pretty => TRUE)),
                    $has_data,
                    "check $test->{name}",
                    {context => 1}
                );
            };
        }
    },
    do_not_die_on_fail => 1,
    fill_databases     => 1,
);

sub convert_block {
    my ($model, $test_data) = @_;
    my $new_block = $model->add(%{$test_data->{block_data}});
    my $pk        = $model->get_pk_fields;
    my $obj       = $model->get($new_block, fields => $pk);

    $model->do_action($obj, "edit", "picategories", $test_data->{picategories},
        "geo", $test_data->{geo}, "brands", $test_data->{brands});

    my $object_before = $model->get($obj, fields => ['*']);
    my $fields_before = {hash_transform($object_before, $test_data->{cpm_fields})};
    my $bk_data_before = get_bk_data($model, $obj)->{$obj->{id}};

    $model->convert_block_cpm_currency(clone($object_before), 'USD', 75);

    my $object_after = $model->get($obj, fields => ['*']);
    my $bk_data_after = get_bk_data($model, $obj)->{$obj->{id}};
    my $fields_after = clone {hash_transform($object_after, $test_data->{cpm_fields})};

    $model->do_action($obj, "edit", %{clone $fields_before});
    $model->do_action($obj, "edit", "is_custom_bk_data" => 1, "bk_data" => to_json($bk_data_before));

    $model->convert_block_cpm_currency(
        clone({%$object_before, "is_custom_bk_data" => 1, "bk_data" => to_json($bk_data_before)}),
        'USD', 75);

    $object_after = $model->get($obj, fields => ['*']);
    my $bk_data_godmode_after = get_bk_data($model, $obj)->{$obj->{id}};

    return {
        fields_before  => $fields_before,
        fields_after   => $fields_after,
        bk_data_before => $bk_data_before,
        bk_data_after  => $bk_data_after,
        godmode_before => $bk_data_before,
        godmode_after  => $bk_data_godmode_after,
    };
}
