#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use File::Path qw(make_path);
use Test::Differences;
use Test::More tests => 1;

use Test::Partner::Utils qw(get_test_data_and_update_if_needed);
use Test::Partner2::Utils qw($SKIP_MODELS);
use Test::Partner2::Mock qw(mock_subs restore_subs);
use Test::Partner2::Simple;

use qbit;

my $OTHER_ID   = '123456';
my $REQUEST_ID = '654321';

my $empty_object;

run_tests(
    sub {
        my ($app) = @_;

        my @models =
          sort grep {$app->$_->DOES('Application::Model::Role::Has::FieldsModeration')} keys(%{$app->get_models()});
        foreach my $model_name (@models) {
            next if $SKIP_MODELS->{$model_name};

            my $data;
            my $model            = $app->$model_name;
            my $moderated_fields = $model->get_fields_moderated();
            $empty_object = {
                moderation => {map {$_ => {}} keys %$moderated_fields},
                map {
                    $moderated_fields->{$_} eq 'ARRAY'
                      ? (
                        $_                                  => [],
                        $model->get_field_name_approved($_) => [],
                      )
                      : (
                        $_                                  => '',
                        $model->get_field_name_approved($_) => '',
                      )
                  } keys %$moderated_fields
            };

            $data->{incorrect_moderation_data} = make_test($model);

            foreach my $field_name (keys %$moderated_fields) {
                $data->{$field_name . '_unknown_request_id_on_empty_object'} =
                  make_test($model, get_moderation_request());

                $data->{$field_name . '_unknown_request_id_on_not_empty_object'} = make_test(
                    $model,
                    get_moderation_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} =
                          ($moderated_fields->{$field_name} eq 'ARRAY' ? ['unmoderated data'] : 'unmoderated data');
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $OTHER_ID, verdict => 0};
                    }
                );

                $data->{$field_name . '_reject_on_not_approved_object'} = make_test(
                    $model,
                    get_reject_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} =
                          ($moderated_fields->{$field_name} eq 'ARRAY' ? ['unmoderated data'] : 'unmoderated data');
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                    }
                );

                $data->{$field_name . '_reject_on_not_approved_object_1/2'} = make_test(
                    $model,
                    get_reject_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['unmoderated data 1', 'unmoderated data 2'];
                        $obj->{moderation}{$field_name}{'unmoderated data 1'} =
                          {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'unmoderated data 2'} = {request_id => $OTHER_ID, verdict => 0};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');

                $data->{$field_name . '_reject_on_not_approved_object_2/2'} = make_test(
                    $model,
                    get_reject_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['new approved data', 'unmoderated data'];
                        $obj->{moderation}{$field_name}{'new approved data'} = {request_id => $OTHER_ID, verdict => 1};
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');

                $data->{$field_name . '_reject_on_approved_object'} = make_test(
                    $model,
                    get_reject_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} =
                          ($moderated_fields->{$field_name} eq 'ARRAY' ? ['unmoderated data'] : 'unmoderated data');
                        $obj->{$model->get_field_name_approved($field_name)} =
                          ($moderated_fields->{$field_name} eq 'ARRAY' ? ['approved data'] : 'approved data');
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'approved data'}    = {request_id => $OTHER_ID,   verdict => 1};
                    }
                );

                $data->{$field_name . '_reject_on_approved_object_1/2'} = make_test(
                    $model,
                    get_reject_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['unmoderated data 1', 'unmoderated data 2'];
                        $obj->{$model->get_field_name_approved($field_name)} = ['approved data'];
                        $obj->{moderation}{$field_name}{'unmoderated data 1'} =
                          {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'unmoderated data 2'} = {request_id => $OTHER_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'approved data'}      = {request_id => $OTHER_ID, verdict => 1};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');

                $data->{$field_name . '_reject_on_approved_object_2/2'} = make_test(
                    $model,
                    get_reject_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['new approved data', 'unmoderated data'];
                        $obj->{$model->get_field_name_approved($field_name)} = ['approved data'];
                        $obj->{moderation}{$field_name}{'new approved data'} = {request_id => $OTHER_ID, verdict => 1};
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'approved data'}    = {request_id => $OTHER_ID,   verdict => 1};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');

                $data->{$field_name . '_approve_on_not_approved_object'} = make_test(
                    $model,
                    get_approve_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} =
                          ($moderated_fields->{$field_name} eq 'ARRAY' ? ['unmoderated data'] : 'unmoderated data');
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                    }
                );

                $data->{$field_name . '_approve_on_not_approved_object_1/2'} = make_test(
                    $model,
                    get_approve_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['unmoderated data 1', 'unmoderated data 2'];
                        $obj->{moderation}{$field_name}{'unmoderated data 1'} =
                          {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'unmoderated data 2'} = {request_id => $OTHER_ID, verdict => 0};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');

                $data->{$field_name . '_approve_on_not_approved_object_2/2'} = make_test(
                    $model,
                    get_approve_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['new approved data', 'unmoderated data'];
                        $obj->{moderation}{$field_name}{'new approved data'} = {request_id => $OTHER_ID, verdict => 1};
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');

                $data->{$field_name . '_approve_on_approved_object'} = make_test(
                    $model,
                    get_approve_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} =
                          ($moderated_fields->{$field_name} eq 'ARRAY' ? ['unmoderated data'] : 'unmoderated data');
                        $obj->{$model->get_field_name_approved($field_name)} =
                          ($moderated_fields->{$field_name} eq 'ARRAY' ? ['approved data'] : 'approved data');
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'approved data'}    = {request_id => $OTHER_ID,   verdict => 1};
                    }
                );

                $data->{$field_name . '_approve_on_approved_object_1/2'} = make_test(
                    $model,
                    get_approve_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['unmoderated data 1', 'unmoderated data 2'];
                        $obj->{$model->get_field_name_approved($field_name)} = ['approved data'];
                        $obj->{moderation}{$field_name}{'unmoderated data 1'} =
                          {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'unmoderated data 2'} = {request_id => $OTHER_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'approved data'}      = {request_id => $OTHER_ID, verdict => 1};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');

                $data->{$field_name . '_approve_on_approved_object_2/2'} = make_test(
                    $model,
                    get_approve_request(),
                    sub {
                        my ($obj) = @_;

                        $obj->{$field_name} = ['new approved data', 'unmoderated data'];
                        $obj->{$model->get_field_name_approved($field_name)} = ['approved data'];
                        $obj->{moderation}{$field_name}{'new approved data'} = {request_id => $OTHER_ID, verdict => 1};
                        $obj->{moderation}{$field_name}{'unmoderated data'} = {request_id => $REQUEST_ID, verdict => 0};
                        $obj->{moderation}{$field_name}{'approved data'}    = {request_id => $OTHER_ID,   verdict => 1};
                    }
                ) if ($moderated_fields->{$field_name} eq 'ARRAY');
            }
            my $expected_data = get_test_data_and_update_if_needed("$model_name.json", $data);
            eq_or_diff($data, $expected_data, $model_name);
        }
    },
    do_not_die_on_fail   => 1,
    dont_create_database => 1,
    fill_databases       => 0,
    locale               => 'C',
);

sub make_test {
    my ($model, $moderation_data, $modifier) = @_;

    my @warn_messages;
    my $obj = clone($empty_object);
    $modifier->($obj) if $modifier;

    mock_subs(
        {
            'Application::Model::Role::Has::FieldsModeration::WARN' => sub {
                my ($data) = @_;

                push @warn_messages, (ref $data ? $data->{message} : $data);
            },
        }
    );

    my $result = $model->_apply_verdict($obj, $moderation_data);
    $result->{warn_messages} = \@warn_messages if scalar @warn_messages;

    restore_subs(['Application::Model::Role::Has::FieldsModeration::WARN']);

    return $result;
}

sub get_moderation_request {
    my ($verdict) = @_;

    return {
        request_id => 654321,
        verdict    => ($verdict // FALSE),
    };
}

sub get_approve_request {
    return get_moderation_request(TRUE);
}

sub get_reject_request {
    return get_moderation_request(FALSE);
}
