#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use File::Path qw(make_path);
use Test::Differences;
use Test::More tests => 1;

use Test::Partner::Utils qw(get_test_data_and_update_if_needed);
use Test::Partner2::Utils qw($SKIP_MODELS);
use Test::Partner2::Mock qw(mock_subs restore_subs);
use Test::Partner2::Simple;

use qbit;

run_tests(
    sub {
        my ($app) = @_;

        my @models =
          sort grep {$app->$_->DOES('Application::Model::Role::Has::FieldsModeration')} keys(%{$app->get_models()});
        foreach my $model_name (@models) {
            next if $SKIP_MODELS->{$model_name};

            my $data;
            my $model = $app->$model_name;

            $data->{verdict_not_found} = make_test($model, {is_verdict_found => FALSE,});
            $data->{normal_verdict_but_not_enough} = make_test(
                $model,
                {
                    is_verdict_found => TRUE,
                    to_save          => {field => 'value',},
                    verdicts         => [1, 1, 1],             # есть непромодерированные
                }
            );
            $data->{reject} = make_test(
                $model,
                {
                    is_verdict_found => TRUE,
                    to_save          => {field => 'value',},
                    verdicts         => [0, 2, 1],             # есть отклонённые
                }
            );
            $data->{approve} = make_test(
                $model,
                {
                    is_verdict_found => TRUE,
                    to_save          => {field => 'value',},
                    verdicts         => [0, 3, 0],             # все приняты
                }
            );
            my $expected_data = get_test_data_and_update_if_needed("$model_name.json", $data);
            eq_or_diff($data, $expected_data, $model_name);
        }
    },
    do_not_die_on_fail   => 1,
    dont_create_database => 1,
    fill_databases       => 0,
    locale               => 'C',
);

sub make_test {
    my ($model, $apply_verdict) = @_;

    my $result = {};

    mock_subs(
        {
            'QBit::Application::Model::Multistate::DB::_get_object_fields' => sub {
                my ($self, $object, $fields, %opts) = @_;

                $result->{fields} = $fields;

                return $object;
            },
            'QBit::Application::Model::Multistate::DB::do_action' => sub {
                my ($self, $object, $action, %opts) = @_;

                $result->{do_action} = {
                    action => $action,
                    opts   => \%opts,
                };

                return $object;
            },
            'QBit::Application::Model::Multistate::DB::maybe_do_action' => sub {
                my ($self, $object, $action, %opts) = @_;

                $result->{maybe_do_action} = {
                    action => $action,
                    opts   => \%opts,
                };

                return $object;
            },
            'Application::Model::Block::Dooh::_apply_verdict' => sub {
                return $apply_verdict;
            },
            'Application::Model::Role::Has::FieldsModeration::WARN' => sub {
                my ($data) = @_;

                push @{$result->{warn_messages}}, (ref $data ? $data->{message} : $data);
            },
        }
    );
    my $obj = {page_id => '12345', id => '123',};
    $result->{result} = $model->on_verdict_received($obj, {'mocked' => 'data'});

    restore_subs(
        [
            'QBit::Application::Model::Multistate::DB::_get_object_fields',
            'QBit::Application::Model::Multistate::DB::do_action',
            'QBit::Application::Model::Multistate::DB::maybe_do_action',
            'Application::Model::Block::Dooh::_apply_verdict',
            'Application::Model::Role::Has::FieldsModeration::WARN',
        ]
    );

    return $result;
}
