#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::More;
use Test::Deep;

use qbit;

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs);

use PiConstants qw($USE_JAVA_JSONAPI_KEY $MAX_PAGE_SIZE);

my $EXPECTED_PROXY_OPTS;
my $MOCK_PROXY_RESULT;

my $EXPECTED_MODEL_OPTS;
my $MOCK_MODEL_RESULT;

my $USE_MODEL = TRUE;

my ($INFO, $WARN, $ERROR);

run_tests(
    sub {
        my ($app) = @_;

        # To see all errors
        local $ENV{TEAMCITY_BUILD_ID} = undef;

        mock_subs(
            {
                'Application::Model::API::Yandex::JavaJsonApi::get_all' => sub {
                    my ($self, $opts) = @_;

                    ok(!$USE_MODEL, 'Use proxy');

                    cmp_deeply($opts, $EXPECTED_PROXY_OPTS, 'check opts for proxy');

                    return $MOCK_PROXY_RESULT;
                },
                'QBit::Application::Model::DBManager::get_all' => sub {
                    my ($self, %opts) = @_;

                    ok($USE_MODEL, 'Use model');

                    cmp_deeply(\%opts, $EXPECTED_MODEL_OPTS, 'check opts for perl model');

                    return $MOCK_MODEL_RESULT;
                },
                'Application::Model::Role::JavaJsonApiProxy::INFO' => sub {
                    my (@args) = @_;

                    $INFO = $args[0];
                },
                'Application::Model::Role::JavaJsonApiProxy::WARN' => sub {
                    my (@args) = @_;

                    $WARN = $args[0];
                },
                'Application::Model::Role::JavaJsonApiProxy::ERROR' => sub {
                    my (@args) = @_;

                    $ERROR = $args[0];
                },
            }
        );

        #default use perl
        $app->kv_store->set($USE_JAVA_JSONAPI_KEY, '{"context_on_site_rtb":false}');
        check_get_from_perl_model($app);

        #use proxy
        $app->kv_store->set($USE_JAVA_JSONAPI_KEY, '{"context_on_site_rtb":true}');

        check_get_from_proxy($app);

        check_filter_and_use_perl_model($app);

        check_no_api_fields_and_use_perl_model($app);

        check_no_fields_and_use_perl_model($app);

        check_public_id_field_is_allowed($app);

        check_not_allowed_options_and_use_perl_model($app);

        check_limit_and_use_perl_model($app);

        get_items_more_than_max_page_size($app);

        catch_exception_and_use_perl_model($app);

    },
    init => [qw(context_on_site_rtb api_java_jsonapi)],
);

sub check_get_from_perl_model {
    my ($app) = @_;

    $EXPECTED_MODEL_OPTS = {fields => [qw(id caption)]};
    $MOCK_MODEL_RESULT = [];
    $app->context_on_site_rtb->get_all(%$EXPECTED_MODEL_OPTS);
}

sub check_get_from_proxy {
    my ($app) = @_;

    $USE_MODEL = FALSE;

    $EXPECTED_PROXY_OPTS = {
        'fields' => ['caption', 'design_templates', 'id', 'is_custom_bk_data'],
        'filter' => {'id' => 1},
        'meta'   => 'total',
        'order_by' => [['id', 0], ['caption', 1]],
        'page_number' => 2,
        'page_size'   => 100,
        'resource'    => 'context_on_site_rtb',
        'user_id'     => 155209804
    };

    $MOCK_PROXY_RESULT = {
        data => [
            {
                attributes => {
                    id               => 1,
                    caption          => 'Test',
                    is_working       => JSON::XS::true,
                    design_templates => [{settings => {hasBorder => JSON::XS::true,},}],
                }
            }
        ]
    };

    my $data = $app->context_on_site_rtb->get_all(
        fields    => [qw(id caption is_custom_bk_data design_templates)],
        filter    => {id => 1},
        order_by  => ['id', ['caption', 1]],
        offset    => 100,
        limit     => 100,
        calc_rows => TRUE
    );

    my $expected = $MOCK_PROXY_RESULT->{'data'}[0];
    $expected->{'attributes'}{'is_custom_bk_data'} = 1;

    cmp_deeply($data, [$expected->{'attributes'}], 'correct convert boolean to number');
}

sub check_filter_and_use_perl_model {
    my ($app) = @_;

    $USE_MODEL = TRUE;

    $EXPECTED_MODEL_OPTS = {fields => [qw(id caption)], filter => ignore()};
    $MOCK_MODEL_RESULT = [];

    $app->context_on_site_rtb->get_all(
        fields => [qw(id caption)],
        filter => $app->partner_db->filter(['AND' => [['id', '=', \1]]])
    );

    ok($INFO =~ /got filter for DB/, 'Correct info message');
}

sub check_no_api_fields_and_use_perl_model {
    my ($app) = @_;

    $USE_MODEL = TRUE;

    $EXPECTED_MODEL_OPTS = {fields => [qw(is_working)]};
    $MOCK_MODEL_RESULT = [];

    $app->context_on_site_rtb->get_all(fields => [qw(is_working)]);

    ok($WARN->{'message'} =~ /got not available fields/, 'Correct info message');
}

sub check_public_id_field_is_allowed {
    my ($app) = @_;

    $USE_MODEL = FALSE;

    $EXPECTED_PROXY_OPTS = {
        'fields'    => [qw(public_id)],
        'page_size' => 1000,
        'resource'  => 'context_on_site_rtb',
        'user_id'   => 155209804
    };
    $MOCK_MODEL_RESULT = [];

    $app->context_on_site_rtb->get_all(fields => [qw(public_id)]);
}

sub check_no_fields_and_use_perl_model {
    my ($app) = @_;

    $USE_MODEL = TRUE;

    $EXPECTED_MODEL_OPTS = {fields => []};
    $MOCK_MODEL_RESULT = [];

    $app->context_on_site_rtb->get_all(fields => []);

    ok($WARN->{'message'} =~ /got empty field set/, 'Correct info message');
}

sub check_not_allowed_options_and_use_perl_model {
    my ($app) = @_;

    $USE_MODEL = TRUE;

    foreach (qw(for_update distinct all_locales group_by)) {
        $EXPECTED_MODEL_OPTS = {fields => [qw(id caption)], $_ => TRUE};
        $MOCK_MODEL_RESULT = [];

        $app->context_on_site_rtb->get_all(%$EXPECTED_MODEL_OPTS);

        ok($WARN->{'message'} =~ /got $_ = 1/, 'Correct info message');
    }
}

sub check_limit_and_use_perl_model {
    my ($app) = @_;

    $USE_MODEL = TRUE;

    $EXPECTED_MODEL_OPTS = {fields => [qw(id caption)], limit => 100500};
    $MOCK_MODEL_RESULT = [];

    $app->context_on_site_rtb->get_all(%$EXPECTED_MODEL_OPTS);

    ok($WARN =~ /got limit more than $MAX_PAGE_SIZE/, 'Correct info message');
}

sub get_items_more_than_max_page_size {
    my ($app) = @_;

    $USE_MODEL = FALSE;

    $EXPECTED_PROXY_OPTS = {
        'fields'    => ['caption', 'id'],
        'page_size' => 1000,
        'resource'  => 'context_on_site_rtb',
        'user_id'   => 155209804
    };

    my $page_number = 0;
    mock_subs(
        {
            'Application::Model::API::Yandex::JavaJsonApi::get_all' => sub {
                my ($self, $opts) = @_;

                ok(!$USE_MODEL, 'Use proxy');

                $page_number++;

                cmp_deeply(
                    $opts,
                    {%$EXPECTED_PROXY_OPTS, ($page_number > 1 ? (page_number => $page_number) : ()),},
                    'check opts for proxy'
                );

                return {data => [$page_number < 5 ? (map {{attributes => {}}} (0 .. $MAX_PAGE_SIZE - 1)) : ()]};
            },
        },
        force => TRUE
    );

    $app->context_on_site_rtb->get_all(fields => [qw(id caption)]);

    ok($WARN =~ /called without limit, but response has $MAX_PAGE_SIZE items/, 'Correct info message');

    ok($ERROR->{'message'} =~ /called without limit, and got more than 5 pages/, 'Correct info message');
}

sub catch_exception_and_use_perl_model {
    my ($app) = @_;

    $USE_MODEL = TRUE;

    $EXPECTED_PROXY_OPTS = {
        'fields'    => ['caption', 'id'],
        'page_size' => 1000,
        'resource'  => 'context_on_site_rtb',
        'user_id'   => 155209804
    };

    $EXPECTED_MODEL_OPTS = {fields => [qw(id caption)]};
    $MOCK_MODEL_RESULT = [];

    mock_subs(
        {
            'Application::Model::API::Yandex::JavaJsonApi::get_all' => sub {
                my ($self, $opts) = @_;

                cmp_deeply($opts, $EXPECTED_PROXY_OPTS, 'check opts for proxy');

                throw Exception('Die');
            },
        },
        force => TRUE
    );

    $app->context_on_site_rtb->get_all(%$EXPECTED_MODEL_OPTS);

    ok($ERROR->{'exception'}->message =~ /Die/, 'Correct info message');
}
