#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use PiConstants qw($SITE_PARTNER_ROLE_ID);

use Test::Partner2::Simple;

use Test::Most;
use Test::Deep;

use qbit;

my $fixture = {
    'site::domain' => {
        'correct and reachable domain - correct' => {
            data   => {'domain' => 'xxx.ru',},
            errors => {},
        },
        'correct but unreachable domain - fail' => {
            data   => {'domain' => 'i_am_a_domain_trust_me.ru',},
            errors => {'domain' => ["Domain doesn't exist"]},
        },
        'undef domain - fail' => {
            data   => {'domain' => undef,},
            errors => {'domain' => ['Data must be defined']},
        },
        'empty string - fail' => {
            data   => {'domain' => '',},
            errors => {'domain' => ['Length "" less than "4"']},
        },
        'domain too short - fail' => {
            data   => {'domain' => '1.c',},
            errors => {'domain' => ['Length "1.c" less than "4"']},
        },
        'domain too long - fail' => {
            data   => {'domain' => 'x' x 255 . '.ru',},
            errors => {'domain' => ['Length "' . 'x' x 255 . '.ru' . '" more than "255"']},
        },
        'correct but yndx .ru domain - fail' => {
            data   => {'domain' => 'music.yandex.ru',},
            errors => {'domain' => ['Don\'t add domain "music.yandex.ru"']},
        },
        'correct but yndx .com domain - fail' => {
            data   => {'domain' => 'music.yandex.com',},
            errors => {'domain' => ['Don\'t add domain "music.yandex.com"']},
        },
        'correct but yndx .com.tr domain - fail' => {
            data   => {'domain' => 'music.yandex.com.tr',},
            errors => {'domain' => ['Don\'t add domain "music.yandex.com.tr"']},
        },
    },
    'owner_site::user_id' => {
        ' user has partner role - correct' => {
            data   => {'user_id' => 9},
            errors => {},
        },
        ' user has no partner role - fail' => {
            data   => {'user_id' => 10},
            errors => {'user_id' => ['Username must belong to the partner']},
        },
        ' user-domain already linked - fail' => {
            data   => {'user_id' => 8},
            errors => {'user_id' => ['You have site with this domain "y.ru"']}
        },
    }
};

run_tests(
    sub {
        my ($app) = @_;

        $app->api_http_gozora;
        $app->{'api_http_gozora'} = Test::MockObject::Extends->new($app->{'api_http_gozora'});
        $app->api_http_gozora->mock('is_site_working', sub {return ($_[1] =~ /xxx/)});

        $app->site;
        $app->{'site'} = Test::MockObject::Extends->new($app->{'site'});
        $app->site->mock('check_rights', sub {return FALSE if 'site_add_yandex_subdomains' eq $_[1]});

        $app->rbac;
        $app->{'rbac'} = Test::MockObject::Extends->new($app->{'rbac'});
        $app->rbac->mock(
            'get_roles_by_user_id',
            sub {
                return {$SITE_PARTNER_ROLE_ID => FALSE || (10 > $_[1]),};
            }
        );

        $app->owner_site;
        $app->{'owner_site'} = Test::MockObject::Extends->new($app->{'owner_site'});
        $app->owner_site->mock(
            'get',
            sub {
                return (9 == $_[1]->{user_id})
                  ? undef
                  : {
                    user_id => $_[1]->{user_id},
                    site    => {domain => 'y.ru'}
                  };
            }
        );

        foreach my $accessor__test_name (sort keys %$fixture) {

            my ($accessor, $test_name) = split('::', $accessor__test_name);

            subtest $test_name => sub {

                foreach my $sub_test_name (sort keys %{$fixture->{$accessor__test_name}}) {
                    my ($errors, $data) = @{$fixture->{$accessor__test_name}->{$sub_test_name}}{qw( errors data  )};

                    _check_validation($app->$accessor, $data, $errors, $test_name . '.' . $sub_test_name);
                }
              }
        }

    },
    do_not_die_on_fail => 1,
    fill_databases     => 0,
);

sub _check_validation {
    my ($model, $data, $expect_errors, $test_name) = @_;

    my $qv = QBit::Validator->new(
        data => $data,
        app  => $model,
        $model->get_template(fields => [keys %$data]),
    );

    my @errors     = $qv->get_fields_with_error();
    my $got_errors = _get_errors(\@errors);

    my $is_ok = FALSE;
    if (%$expect_errors) {
        $is_ok = cmp_deeply($got_errors, $expect_errors, $test_name);
    } else {
        $is_ok = is($qv->has_errors, FALSE, $test_name);
    }

    warn Data::Dumper->Dump([$got_errors], ['$got_errors ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;

    return 1;
}

sub _get_errors {
    my ($errors) = @_;

    my $res = {
        # <path> => [ 'msg1', ... ]
    };
    foreach my $row (@$errors) {
        my $key = join '.', @{$row->{path}};
        my $ar = $res->{$key} //= [];
        push @$ar, @{$row->{msgs}};
    }

    return $res;
}
