use qbit;

use Test::Partner2::Simple;
use Test::More;
use File::Temp qw(tempdir);
use File::Slurp qw(write_file);
use File::Basename;

run_tests(
    sub {
        my ($app) = @_;

        my ($local_cache_dir, $remote_cache_dir) = initialize($app);

        my $data = $app->stat_download_data->get_all(fields => [qw(method sync)]);
        my @local_files  = map {basename($_)} glob("$local_cache_dir/*");
        my @remote_files = map {basename($_)} glob("$remote_cache_dir/*");

        is_deeply(
            $data,
            [
                {
                    'sync'   => '1',
                    'method' => 'synced_local'
                },
                {
                    'sync'   => '1',
                    'method' => 'synced_remote'
                },
                {
                    'sync'   => '0',
                    'method' => 'unsynced_local'
                },
                {
                    'sync'   => '0',
                    'method' => 'unsynced_remote'
                },
                {
                    'sync'   => '0',
                    'method' => 'deleted_local'
                },
                {
                    'sync'   => '0',
                    'method' => 'deleted_remote'
                }
            ],
            'inital stat_download_data'
        );
        is_deeply(
            \@local_files,
            [
                'deleted_remote_2018-01-25.dump', 'synced_local_2018-01-25.dump',
                'synced_remote_2018-01-25.dump',  'unsynced_local_2018-01-25.dump'
            ],
            'inital local_files'
        );
        is_deeply(
            \@remote_files,
            [
                'deleted_local_2018-01-25.dump', 'synced_local_2018-01-25.dump',
                'synced_remote_2018-01-25.dump', 'unsynced_remote_2018-01-25.dump'
            ],
            'initial remote files'
        );

        $app->stat_download_data->pull_changes_from_other_host();

        $data = $app->stat_download_data->get_all(fields => [qw(method sync)], order_by => [qw(method)]);
        @local_files  = map {basename($_)} glob("$local_cache_dir/*");
        @remote_files = map {basename($_)} glob("$remote_cache_dir/*");

        is_deeply(
            $data,
            [
                {
                    'sync'   => '0',
                    'method' => 'deleted_local'
                },
                {
                    'sync'   => '1',
                    'method' => 'synced_local'
                },
                {
                    'sync'   => '1',
                    'method' => 'synced_remote'
                },
                {
                    'sync'   => '0',
                    'method' => 'unsynced_local'
                },
                {
                    'sync'   => '1',
                    'method' => 'unsynced_remote'
                },
            ],
            'stat_download_data after pull'
        );
        is_deeply(
            \@local_files,
            [
                'synced_local_2018-01-25.dump',   'synced_remote_2018-01-25.dump',
                'unsynced_local_2018-01-25.dump', 'unsynced_remote_2018-01-25.dump'
            ],
            'local files after pull'
        );
        is_deeply(
            \@remote_files,
            [
                'deleted_local_2018-01-25.dump', 'synced_local_2018-01-25.dump',
                'synced_remote_2018-01-25.dump', 'unsynced_remote_2018-01-25.dump'
            ],
            'remote_files after pull'
        );
    },
    fill_databases => 0,
);

sub initialize {
    my ($app) = @_;

    my $local_dir  = tempdir('local_XXXXXX',  TMPDIR => 1);
    my $remote_dir = tempdir('remote_XXXXXX', TMPDIR => 1);
    my $local_cache_dir  = "$local_dir/stat_cache";
    my $remote_cache_dir = "$remote_dir/stat_cache";

    mkdir $local_cache_dir  or die "mkdir: $!";
    mkdir $remote_cache_dir or die "mkdir: $!";

    $app->set_option(hostname => $local_dir);
    $app->stat_download_data->set_option(local_stat_cache_dir  => "$local_dir/stat_cache/");
    $app->stat_download_data->set_option(remote_stat_cache_dir => '/stat_cache/');

    add_files(
        app   => $app,
        dirs  => [$local_cache_dir, $remote_cache_dir,],
        files => [
            {
                method      => 'synced_local',
                stat_date   => '2018-01-25',
                hostname    => $local_dir,
                checksum    => 'fake_checksum',
                update_time => '2018-01-25',
                filename    => 'synced_local_2018-01-25.dump',
                sync        => 1,
            },
            {
                method      => 'synced_remote',
                stat_date   => '2018-01-25',
                hostname    => $remote_dir,
                checksum    => 'fake_checksum',
                update_time => '2018-01-25',
                filename    => 'synced_remote_2018-01-25.dump',
                sync        => 1,
            },
        ]
    );

    add_files(
        app   => $app,
        dirs  => [$local_cache_dir,],
        files => [
            {
                method      => 'unsynced_local',
                stat_date   => '2018-01-25',
                hostname    => $local_dir,
                checksum    => 'fake_checksum',
                update_time => '2018-01-25',
                filename    => 'unsynced_local_2018-01-25.dump',
                sync        => 0,
            },
        ]
    );

    add_files(
        app   => $app,
        dirs  => [$remote_cache_dir,],
        files => [
            {
                method      => 'unsynced_remote',
                stat_date   => '2018-01-25',
                hostname    => $remote_dir,
                checksum    => 'fake_checksum',
                update_time => '2018-01-25',
                filename    => 'unsynced_remote_2018-01-25.dump',
                sync        => 0,
            },
        ]
    );

    add_files(
        app   => $app,
        dirs  => [$remote_cache_dir,],
        files => [
            {
                method      => 'deleted_local',
                stat_date   => '2018-01-25',
                hostname    => $local_dir,
                checksum    => 'fake_checksum',
                update_time => '2018-01-25',
                filename    => 'deleted_local_2018-01-25.dump',
                sync        => 0,
            },
        ]
    );

    add_files(
        app   => $app,
        dirs  => [$local_cache_dir,],
        files => [
            {
                method      => 'deleted_remote',
                stat_date   => '2018-01-25',
                hostname    => $remote_dir,
                checksum    => 'fake_checksum',
                update_time => '2018-01-25',
                filename    => 'deleted_remote_2018-01-25.dump',
                sync        => 0,
            },
        ]
    );

    return ($local_cache_dir, $remote_cache_dir);
}

sub add_files {
    my (%opts) = @_;

    for my $file (@{$opts{files}}) {
        for my $dir (@{$opts{dirs}}) {
            write_file("$dir/$file->{filename}", "$file->{filename}_contents");
        }

        $opts{app}->partner_db->stat_download_data->add($file);
    }
}
