use strict;
use warnings FATAL => 'all';

use Test::More;
use Test::Differences qw(eq_or_diff);

use PiConstants qw($XLS_LIMIT  $XLSX_LIMIT);

use qbit;
use Test::Partner2::Simple;
use Test::Partner::Utils qw(get_test_data_and_update_if_needed mock_curdate);

my $FIXTURE = {
    'case #1' => {
        rows_limit  => undef,
        digest_data => {
            id     => 'hash#1',
            params => to_json(
                {
                    chartLevel => undef,
                    tableLevel => [
                        {
                            dimension_fields => ['date|day'],
                            dimension_filter => undef,
                            entity_fields    => [],
                            fields           => ['shows', 'clicks_direct'],
                            levels           => [
                                {
                                    filter => undef,
                                    id     => 'payment'
                                }
                            ],
                            limits => {
                                limit  => 50,
                                offset => 0
                            },
                            order_by => [],
                            period   => ['2017-06-02', '2017-06-03'],
                            total    => 0,
                            vat      => -1
                        }
                    ],
                    'tableViewModelData' => {
                        sortBy    => 'date',
                        sortOrder => 'asc'
                    }
                }
            ),
        },
        mol_mock_data => {
            "header" => ["UpdateTime", "DirectBannerClicks", "RenderShows"],
            "data" => [["2017-06-02", 4120779, 849070208], ["2017-06-03", 3549202, 693429508]],
            "totals" => {
                "RenderShows"        => 1542499716,
                "DirectBannerClicks" => 7669981
            },
        },
    },
    'case #2' => {
        rows_limit  => $XLSX_LIMIT,
        digest_data => {
            id     => 'hash#2',
            params => to_json(
                {
                    chartLevel => undef,
                    tableLevel => [
                        {
                            dimension_fields => [],
                            dimension_filter => undef,
                            entity_fields    => [],
                            fields           => ['partner_wo_nds',],
                            levels           => [
                                {
                                    filter => undef,
                                    id     => 'payment',
                                },
                            ],
                            limits => {
                                limit  => 50,
                                offset => 0,
                            },
                            order_by => [],
                            period   => 'today',
                            total    => 0,
                            vat      => -1,
                        }
                    ],
                    tableViewModelData => {
                        sortBy    => 'partner_wo_nds',
                        sortOrder => 'desc',
                    },
                }
            ),
        },
        mol_mock_data => {
            "header" => ["MixedPartnerPrice"],
            "data"   => [[47000000]]
        },
    },
    'no limit query' => {
        rows_limit  => undef,
        digest_data => {
            id     => 'hash#3',
            params => to_json(
                {
                    chartLevel => undef,
                    tableLevel => [
                        {
                            dimension_fields => [],
                            dimension_filter => undef,
                            entity_fields    => [],
                            fields           => ['partner_wo_nds',],
                            levels           => [
                                {
                                    filter => undef,
                                    id     => 'payment',
                                },
                            ],
                            limits => {
                                limit  => 50,
                                offset => 0,
                            },
                            order_by => [],
                            period   => 'today',
                            total    => 0,
                            vat      => -1,
                        }
                    ],
                    tableViewModelData => {
                        sortBy    => 'partner_wo_nds',
                        sortOrder => 'desc',
                    },
                }
            ),
        },
        mol_mock_data => {
            "header" => ["MixedPartnerPrice"],
            "data"   => [[47000000]]
        },
    },
};

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate('2020-02-18 18:28:12');

        change_cur_user($app, 'yndx-developer');

        my $got_data = {};
        foreach my $case_name (sort keys %$FIXTURE) {
            my ($digest_data, $mol_mock_data, $rows_limit, $totals) =
              @{$FIXTURE->{$case_name}}{qw( digest_data  mol_mock_data  rows_limit totals )};

            $app->partner_db->stat_report_params_digest->add($digest_data);

            my $mol_mock_content = {
                'success'  => 1,
                'protocol' => 'HTTP/1.1',
                'headers'  => {
                    'connection'        => 'close',
                    'content-type'      => 'text/plain; charset=utf-8',
                    'transfer-encoding' => 'chunked',
                    'date'              => 'Tue, 27 Jun 2017 15:29:46 GMT',
                    'vary'              => 'Accept-Encoding',
                    'server'            => 'nginx/1.8.0'
                },
                'status'  => '200',
                'content' => to_json(
                    {
                        "status"          => 0,
                        "error_text"      => "",
                        "stat_time"       => 20170627181422,
                        "duration"        => 3.933,
                        "host-aggregator" => "bsfastexport02i.yabs.yandex.ru",
                        "header"          => $mol_mock_data->{header},
                        "total_rows"      => scalar(@{$mol_mock_data->{data}}),
                        "data"            => $mol_mock_data->{data},
                        "totals"          => $mol_mock_data->{totals},
                    }
                ),
                'reason' => 'OK'
            };

            my $mol_request;

            no strict 'refs';
            no warnings 'redefine';
            local *{'HTTP::Tiny::request'} = sub {
                my ($self, $method, $url, $args) = @_;
                $mol_request = from_json($args->{content});
                return $mol_mock_content;
            };

            my $mol_data = $app->statistics->get_data_from_hash2(
                report_hash => $digest_data->{id},
                rows_limit  => $rows_limit
            );

            fix_numbers($mol_data->{sorted_stat_data});

            fix_numbers($mol_data->{sorted_stat_data});

            $got_data->{$case_name} = {
                mol_request => $mol_request,
                mol_data    => $mol_data,
            };
        }

        my $expected = get_test_data_and_update_if_needed('data.json', $got_data);
        eq_or_diff($got_data, $expected, 'check data', {context => 3});
    },
);

sub fix_numbers {
    my ($data) = @_;
    foreach my $row (@$data) {
        map {$row->{$_} += 0}
          grep {defined $row->{$_} && $row->{$_} ne '-'} qw(partner_wo_nds);
    }
}
