#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::More;
use Test::Deep;
use Test::Partner::Utils;

use qbit;

my $FIRST_SYSTEM_EVENT;
my $FIRST_START_DT = '2021-02-11 18:51:34';
my $FIRST_STOP_DT  = '2021-02-11 18:53:12';

my $SECOND_SYSTEM_EVENT;
my $SECOND_START_DT = '2021-02-11 18:52:24';
my $SECOND_STOP_DT  = '2021-02-11 18:54:41';

run_tests(
    sub {
        my ($app) = @_;

        $FIRST_SYSTEM_EVENT = {
            'details' => {
                'hostname' => $app->get_option('hostname'),
                'pid'      => $$,
                'user'     => $ENV{'USER'},
            },
            'id'       => ignore(),
            'start_dt' => $FIRST_START_DT,
            'stop_dt'  => undef,
            'type'     => 'first'
        };

        $SECOND_SYSTEM_EVENT = {
            'details' => {
                'hostname' => $app->get_option('hostname'),
                'pid'      => $$,
                'user'     => $ENV{'USER'},
                'ticket'   => 'PI-12345',
            },
            'id'       => ignore(),
            'start_dt' => $SECOND_START_DT,
            'stop_dt'  => undef,
            'type'     => 'second'
        };

        start_first_event($app);

        start_second_event($app);

        check_action_log($app);

        stop_first_event($app);

        stop_second_event_in_post_run($app);
    },
);

sub start_first_event {
    my ($app) = @_;

    mock_curdate($FIRST_START_DT);

    my $first_system_event = $app->system_events->start('first');

    cmp_deeply($first_system_event, $FIRST_SYSTEM_EVENT, 'first system event');

    my $system_events = $app->system_events->get_all();

    cmp_deeply($system_events, [$FIRST_SYSTEM_EVENT], 'current system events - only first');

    my $db_first_system_event = $app->partner_db->system_events->get($first_system_event->{'id'});
    $db_first_system_event->{'details'} = to_json(from_json($db_first_system_event->{'details'}), pretty => TRUE);

    cmp_deeply($db_first_system_event,
        {%$FIRST_SYSTEM_EVENT, details => to_json($FIRST_SYSTEM_EVENT->{'details'}, pretty => TRUE),});
}

sub start_second_event {
    my ($app) = @_;

    mock_curdate($SECOND_START_DT);

    my $second_system_event = $app->system_events->start('second', ticket => 'PI-12345');

    cmp_deeply($second_system_event, $SECOND_SYSTEM_EVENT, 'second system event');

    my $system_events = $app->system_events->get_all();

    cmp_deeply($system_events, [$FIRST_SYSTEM_EVENT, $SECOND_SYSTEM_EVENT], 'current system events');

    my $db_second_system_event = $app->partner_db->system_events->get($second_system_event->{'id'});
    $db_second_system_event->{'details'} = to_json(from_json($db_second_system_event->{'details'}), pretty => TRUE);

    cmp_deeply($db_second_system_event,
        {%$SECOND_SYSTEM_EVENT, details => to_json($SECOND_SYSTEM_EVENT->{'details'}, pretty => TRUE),});
}

sub check_action_log {
    my ($app) = @_;

    my $rtb = $app->context_on_site_rtb->get_all(
        fields => ['campaign_id', 'id'],
        filter => {multistate => 'working'},
        limit  => 1
    )->[0];

    ok(defined($rtb), 'Found rtb block');

    my $action_logs = $app->partner_db->context_on_site_rtb_action_log->get_all(
        fields => [qw(comment)],
        filter => [
            'AND',
            [
                [elem_campaign_id => '='  => \$rtb->{'campaign_id'}],
                [elem_id          => '='  => \$rtb->{'id'}],
                ['comment'        => '<>' => \'']
            ]
        ]
    );

    ok(!@$action_logs, 'logs do not exist');

    use_perl_do_actions($app, 'context_on_site_rtb');
    $app->context_on_site_rtb->do_action($rtb, 'stop');

    $action_logs = $app->partner_db->context_on_site_rtb_action_log->get_all(
        fields => [qw(comment)],
        filter => [
            'AND',
            [
                [elem_campaign_id => '='  => \$rtb->{'campaign_id'}],
                [elem_id          => '='  => \$rtb->{'id'}],
                ['comment'        => '<>' => \'']
            ]
        ]
    );

    ok(@$action_logs, 'logs exist');

    ok($action_logs->[0]{'comment'} eq 'system_event_ids: 1, 2 (PI-12345)', 'comment is correct');
}

sub stop_first_event {
    my ($app) = @_;

    mock_curdate($FIRST_STOP_DT);

    my $system_events = $app->system_events->get_all();

    my ($deleted_system_event) = $app->system_events->stop($system_events->[0]{'id'});

    $system_events = $app->system_events->get_all();

    cmp_deeply($system_events, [$SECOND_SYSTEM_EVENT], 'current system events - only second');

    my $db_deleted_system_event = $app->partner_db->system_events->get($deleted_system_event->{'id'});
    $db_deleted_system_event->{'details'} = to_json(from_json($db_deleted_system_event->{'details'}), pretty => TRUE);

    cmp_deeply(
        $db_deleted_system_event,
        {
            %$FIRST_SYSTEM_EVENT,
            details => to_json($FIRST_SYSTEM_EVENT->{'details'}, pretty => TRUE),
            stop_dt => $FIRST_STOP_DT,
        }
    );
}

sub stop_second_event_in_post_run {
    my ($app) = @_;

    mock_curdate($SECOND_STOP_DT);

    my $second_system_event = $app->system_events->get_all()->[0];

    $app->system_events->post_run();

    my $system_events = $app->system_events->get_all();

    cmp_deeply($system_events, [], 'current system events are empty');

    my $db_second_system_event = $app->partner_db->system_events->get($second_system_event->{'id'});
    $db_second_system_event->{'details'} = to_json(from_json($db_second_system_event->{'details'}), pretty => TRUE);

    cmp_deeply(
        $db_second_system_event,
        {
            %$SECOND_SYSTEM_EVENT,
            details => to_json($SECOND_SYSTEM_EVENT->{'details'}, pretty => TRUE),
            stop_dt => $SECOND_STOP_DT,
        }
    );
}
