use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_bk);
use qbit;

use utf8;
use Test::More;
use Test::Exception;
use Test::Deep qw(cmp_bag);

my $USER_LOGIN = 'mocked-yan-partner';
my $USER_PAGES = [
    # { id => ..,  model => ... },
];

run_tests(
    sub {
        my ($app) = @_;

        mock_bk($app);

        # Get user pages (that must be send to BK)
        {
            my $user_id = $app->users->get_by_login($USER_LOGIN, fields => ['id'])->{'id'};
            my @page_ids =
              map {$_->{'page_id'}} @{$app->cur_user->get_all_page_ids_owned_by_user_for_global_concurents($user_id)};
            $USER_PAGES = $app->all_pages->get_all(
                fields => ['id', 'page_id', 'model'],
                filter => {page_id => \@page_ids}
            );
        }

        $app->set_app_locale('en');

        check_initial_db_state($app);

        # Проверяем что на площадках отсутствует флаг "need_update"
        _check_user_pages_flag_need_update($app, FALSE);

        check_adding_correct_domain($app);

        # Проверяем что на всех площадках выставился флаг "need_update"
        _check_user_pages_flag_need_update($app, TRUE);

        check_adding_duplicate_domain($app);
        check_adding_incorrect_domain($app);

        check_adding_domain_on_deleted_page($app);

    },
    user => $USER_LOGIN,
);

sub check_initial_db_state {
    my ($app) = @_;

    cmp_bag(
        $app->user_global_excluded_domains->get_all_domains_for_cur_user(),
        ['design.ru', 'rambler.ru', 'сайт.рф'],
        'Correct initial database state',
    );

}

sub check_adding_correct_domain {
    my ($app) = @_;

    $app->user_global_excluded_domains->add($app->cur_user->get_cur_user_id(), 'zzz.yandex ');

    cmp_bag(
        $app->user_global_excluded_domains->get_all_domains($app->cur_user->get_cur_user_id()),
        ['design.ru', 'rambler.ru', 'zzz.yandex', 'сайт.рф'],
        'Correct database state after adding new global excluded yandex',
    );
}

sub check_adding_duplicate_domain {
    my ($app) = @_;

    throws_ok {
        $app->user_global_excluded_domains->add($app->cur_user->get_cur_user_id(), 'zzz.yandex');
    }
    qr/Excluded domain "zzz\.yandex" already added/,
      'user_global_excluded_domains->add() dies when trying to add the same domain';
}

sub check_adding_incorrect_domain {
    my ($app) = @_;

    throws_ok {
        $app->user_global_excluded_domains->add($app->cur_user->get_cur_user_id(), '...');
    }
    qr/Invalid domain: "\.\.\."/, 'user_global_excluded_domains->add() dies when trying to add incorrect domain';
}

sub check_adding_domain_on_deleted_page {
    my ($app) = @_;

    my $deleted_page_index = 0;
    my $regular_page_index = 1;
    my ($del_id, $del_model) = @{$USER_PAGES->[$deleted_page_index]}{qw( id model )};
    {
        # Удаляем одну из площадок, чтобы потом проверить на ней флаг "need_update"
        $app->$del_model->do_action({id => $del_id}, 'stop');
        $app->$del_model->do_action({id => $del_id}, 'delete');

        # Делаем отправку в БК чтобы сбросить флаги "need_update" выставленный ранее
        foreach my $object (@$USER_PAGES[$deleted_page_index, $regular_page_index]) {
            my ($id, $model) = @$object{qw( id  model )};
            $app->$model->update_in_bk({id => $id});
        }
    }

    $app->user_global_excluded_domains->add($app->cur_user->get_cur_user_id(), 'foo.yandex');

    {
        # Проверяем что на удаленной площадке НЕ выставился флаг "need_update"
        _check_user_pages_flag_need_update($app, FALSE, $USER_PAGES->[$deleted_page_index]->{'id'});

        # Проверяем что при этом на другой площадке флаг "need_update" выставился
        _check_user_pages_flag_need_update($app, TRUE, $USER_PAGES->[$regular_page_index]->{'id'});
    }
}

sub _check_user_pages_flag_need_update {
    my $app           = shift;
    my $expect        = shift || FALSE;
    my $check_id_only = shift;

    my $no = $expect ? 'got' : 'has no';

    my $expect_status = 'need_update';

    subtest(
        'All user pages has ' . $no . ' flag "need_update"' => sub {
            foreach my $object (@$USER_PAGES) {
                my ($id, $page_id, $model) = @$object{qw( id  page_id  model )};

                if (!$check_id_only || $check_id_only == $id) {
                    my $multiistate = $app->$model->get($id, fields => ['multistate'])->{'multistate'};
                    my $got = $app->$model->check_multistate_flag($multiistate, $expect_status);
                    is(
                        $got, $expect,
                        sprintf(
                            'page_id=%s (%s) %s multistate=%d with state "%s"',
                            $page_id, $model, $no, $multiistate, $expect_status
                        )
                      );
                }
            }
        }
    );
}
