#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::Exception;
use Test::More;
use Test::Deep qw(cmp_deeply);

use Test::Partner2::Mock;
use Test::Partner2::Simple;

use qbit;

use Utils::UniqueID qw(make_unique_id);

my $USER_ID = 1009;

my $PAGE_ID_WITH_WORKING_BLOCKS = 41443;
my $PAGE_ID_WITHOUT_BLOCKS      = 801;

my $WORKING_BLOCK_ID = {
    'id'          => 1,
    'campaign_id' => $PAGE_ID_WITH_WORKING_BLOCKS,
};

run_tests(
    sub {
        my ($app) = @_;

        check_context_on_site_rtb($app);

        check_context_on_site_content($app);
    },
);

sub check_context_on_site_rtb {
    my ($app) = @_;

    #copy working block
    my $rtb_block_to_copy = $app->partner_db->context_on_site_rtb->get($WORKING_BLOCK_ID);
    $rtb_block_to_copy->{'campaign_id'} = $PAGE_ID_WITHOUT_BLOCKS;
    $rtb_block_to_copy->{'unique_id'} =
      make_unique_id('context_on_site_rtb', $PAGE_ID_WITHOUT_BLOCKS, $rtb_block_to_copy->{'id'});
    $app->partner_db->context_on_site_rtb->add($rtb_block_to_copy);

    $app->partner_db->queue->truncate();

    my $ext_rtb_before = $app->context_on_site_rtb->get_all(
        fields => [qw(public_id site_version owner_id)],
        filter => {multistate => 'working'}
    );
    check_ext_rtb_before($ext_rtb_before);

    my $int_rtb_before = $app->internal_context_on_site_rtb->get_all(
        fields => [qw(public_id site_version owner_id)],
        filter => {multistate => 'working'}
    );
    check_int_rtb($int_rtb_before);

    $app->users->_delete_featured_site_versions_blocks($USER_ID, [qw(context_on_site_rtb internal_context_on_site_rtb)],
        [qw(desktop)]);

    my $ext_rtb_after = $app->context_on_site_rtb->get_all(
        fields => [qw(public_id site_version owner_id)],
        filter => {multistate => 'working'}
    );
    check_ext_rtb_after($app, $ext_rtb_after);

    my $int_rtb_after = $app->internal_context_on_site_rtb->get_all(
        fields => [qw(public_id site_version owner_id)],
        filter => {multistate => 'working'}
    );
    check_int_rtb($int_rtb_after);
}

sub check_ext_rtb_before {
    my ($blocks) = @_;

    my @desktop_blocks = grep {$_->{'site_version'} eq 'desktop' && $_->{'owner_id'} == $USER_ID} @$blocks;
    ok(@desktop_blocks, "Working blocks with desktop exist for user $USER_ID");

    cmp_deeply(
        \@desktop_blocks,
        [
            {
                public_id    => "R-A-$PAGE_ID_WITH_WORKING_BLOCKS-1",
                site_version => 'desktop',
                owner_id     => $USER_ID,
            },
            {
                public_id    => "R-A-$PAGE_ID_WITH_WORKING_BLOCKS-100000",
                site_version => 'desktop',
                owner_id     => $USER_ID,
            }
        ],
        'Blocks are correct'
    );

    ok(
        (grep {$_->{'site_version'} ne 'desktop' && $_->{'owner_id'} == $USER_ID} @$blocks),
        "Working blocks with another site version exist for user $USER_ID"
      );

    ok((grep {$_->{'site_version'} eq 'desktop' && $_->{'owner_id'} != $USER_ID} @$blocks),
        "Working blocks with desktop exist for another user");
}

sub check_int_rtb {
    my ($blocks) = @_;

    ok((grep {$_->{'site_version'} eq 'desktop'} @$blocks), "Working blocks with desktop exist");

    ok((grep {$_->{'site_version'} ne 'desktop'} @$blocks), "Working blocks with another site version exist");
}

sub check_ext_rtb_after {
    my ($app, $blocks) = @_;

    ok(
        (grep {$_->{'site_version'} eq 'desktop' && $_->{'owner_id'} == $USER_ID} @$blocks),
        "Working blocks with desktop exist for user $USER_ID, but tasks were created"
      );
    my $queue = $app->queue->get_all(fields => [qw(params_ref)]);

    cmp_deeply(
        [map {$_->{'params_ref'}{'modelIds'}[0]} @$queue],
        [
            make_unique_id('context_on_site_rtb', $PAGE_ID_WITH_WORKING_BLOCKS, 1),
            make_unique_id('context_on_site_rtb', $PAGE_ID_WITH_WORKING_BLOCKS, 100000),
        ],
        'Public ids are correct'
    );

    ok(
        (grep {$_->{'site_version'} ne 'desktop' && $_->{'owner_id'} == $USER_ID} @$blocks),
        "Working blocks with another site version exist for user $USER_ID"
      );

    ok((grep {$_->{'site_version'} eq 'desktop' && $_->{'owner_id'} != $USER_ID} @$blocks),
        "Working blocks with desktop exist for another user");
}

sub check_context_on_site_content {
    my ($app) = @_;

    my $content_blocks_before = $app->context_on_site_content->get_all(
        fields => [qw(public_id)],
        filter => {site_version => 'general', multistate => 'working'}
    );
    ok(@$content_blocks_before, 'Blocks with site version "general" exist');

    $app->users->_delete_featured_site_versions_blocks($USER_ID, [qw(context_on_site_content)], [qw(general)]);

    my $content_blocks_after = $app->context_on_site_content->get_all(
        fields => [qw(public_id)],
        filter => {site_version => 'general', multistate => 'working'}
    );
    ok(!@$content_blocks_after, 'Blocks with site version "general" do not exist');
}
