#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::More tests => 17;
use Test::Exception;
use Test::Differences;
use Test::Partner::Utils;

use qbit;

my @MODELS = qw/
  search_on_site_campaign
  context_on_site_campaign
  mobile_app_settings
  video_an_site
  /;

sub get_active_pages {
    my ($app, $user) = @_;

    my $active_pages;
    foreach my $model (@MODELS) {
        my $pages = $app->$model->get_all(
            fields => [qw(id page_id)],
            filter => {multistate => 'working or testing', owner_id => $user->{id}},
        );
        push @$active_pages, @$pages;
    }

    return $active_pages;
}

run_tests(
    sub {
        my ($app) = @_;

        $app->{'__CURRENT_USER_RIGHTS__'} = {};

        {
            no warnings qw(redefine once);

            *QBit::Application::Model::SendMail::send = sub { };
        }

        my $site_partner = {
            id           => 1009,
            login        => 'mocked-yan-partner',
            role         => 9,
            pages_before => [{id => 799, page_id => 799}, {id => 41443, page_id => 41443}],
            pages_mid    => [],
            pages_after  => [{id => 799, page_id => 799}, {id => 41443, page_id => 41443}],
            try_restore => {model => 'context_on_site_campaign', id => 41443}
        };
        my $video_partner = {
            id           => 1010,
            login        => 'mocked-video-partner',
            role         => 20,
            pages_before => [{id => 801, page_id => 801}, {id => 154689, page_id => 154689}],
            pages_mid => [{id => 801, page_id => 801}],    # context_on_site_campaign page
            pages_after =>
              [{id => 801, page_id => 801}, {id => 128972, page_id => 128972}, {id => 154689, page_id => 154689}],

            try_restore => {model => 'video_an_site', id => 154689}
        };
        my $mobile_partner = {
            id           => 1011,
            login        => 'mocked-mobile-app-partner',
            role         => 27,
            pages_before => [{id => 43569, page_id => 43569}],
            pages_mid    => [],
            pages_after  => [{id => 43569, page_id => 43569}],
            try_restore  => {model => 'mobile_app_settings', id => 43569}
        };
        my $developer = {id => 1022, login => 'yndx-developer'};

        foreach my $user ($site_partner, $video_partner, $mobile_partner) {
            $app->set_cur_user($user);
            eq_or_diff(get_active_pages($app, $user),
                $user->{pages_before}, "Initial state of working pages for $user->{login}");

            $app->set_cur_user($developer);
            $app->users->do_action($user->{id}, 'revoke_roles', roles_id => $user->{role});

            $app->queue->start($app->queue->grab_next_task());

            $app->set_cur_user($user);
            eq_or_diff(get_active_pages($app, $user),
                $user->{pages_mid}, "State of working pages for $user->{login} after role $user->{role} revocation");

            $app->set_cur_user($developer);
            my $model = $user->{try_restore}{model};
            throws_ok(sub {$app->$model->do_action($user->{try_restore}{id}, 'restore')},
                'Exception::Denied', "Can't restore page before assigning role to owner");
            $app->users->do_action($user->{id}, 'set_user_role', role_id => $user->{role});

            $app->set_cur_user($user);
            eq_or_diff(get_active_pages($app, $user),
                $user->{pages_after}, "State of working pages for $user->{login} after role $user->{role} addition");

            $app->set_cur_user($developer);
            $app->users->do_action($user->{id}, 'revoke_roles', roles_id => $user->{role});

            $app->queue->start($app->queue->grab_next_task());

            $app->set_cur_user($user);
            eq_or_diff(get_active_pages($app, $user),
                $user->{pages_mid},
                "State of working pages for $user->{login} after second role $user->{role} revocation");
        }
    },
    init  => [qw(mailer)],
    mocks => [qw(mock_format_system)],
);
