#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Deep;

use qbit;

use Test::More;

use Test::Partner2::Mock qw(mock_curdate);
use Test::Partner2::Simple;
use Test::Partner2::Fixture;

use PiConstants qw($MYSQL_MIN_DATETIME);

my $MODERATION_UPDATE_TIME = '2019-08-20 15:45:00';
my $UPDATE_TIME            = '2019-08-20 15:50:00';
my $MOCKED_DATE            = '2019-08-20 15:55:00';
my $mocked_pages_log_path  = 'mocked_pages_log_path';
my $mocked_blocks_log_path = 'mocked_blocks_log_path';

my @up_to_date_page_fixtures  = ('partner_indoor_page',                'partner_outdoor_page',);
my @up_to_date_block_fixtures = ('partner_indoor_video_block',         'partner_outdoor_video_block',);
my @updated_page_fixtures     = ('partner_updated_indoor_page',        'partner_updated_outdoor_page',);
my @updated_block_fixtures    = ('partner_updated_indoor_video_block', 'partner_updated_outdoor_video_block',);
my @fixtures =
  (@up_to_date_page_fixtures, @up_to_date_block_fixtures, @updated_page_fixtures, @updated_block_fixtures,);

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate($MOCKED_DATE);

        no strict 'refs';
        no warnings 'redefine';

        my $fixtures_hash = get_fixtures_hash(@fixtures);

        my $paths = $app->get_option('logbroker');
        $paths->{pages_topic}  = $mocked_pages_log_path;
        $paths->{blocks_topic} = $mocked_blocks_log_path;
        $app->set_option('logbroker', $paths);

        my $methods = $app->get_cron_methods();

        my $method = $methods->{'adv_net_dooh'}{'send_updated_items_to_logbroker'}{'package'}->new(app => $app);

        $app->kv_store->set($method->get_moderation_update_time_key('indoor'),  $MODERATION_UPDATE_TIME);
        $app->kv_store->set($method->get_moderation_update_time_key('outdoor'), $MODERATION_UPDATE_TIME);

        my @got_pages;
        my @got_blocks;

        *{'Application::Model::API::Yandex::SelfService::logbroker'} = sub {
            my ($self, %opts) = @_;
            my ($file, $data) = @opts{qw(topic data)};
            if ($file eq $mocked_pages_log_path) {
                push @got_pages, @$data;
            } elsif ($file eq $mocked_blocks_log_path) {
                push @got_blocks, @$data;
            } else {
                fail("Unexpected topic: $file");
            }
        };

        $app->do('adv_net_dooh', 'send_updated_items_to_logbroker');

        my @expected_pages = (
            {
                'update_time' => $UPDATE_TIME,
                'model'       => 'indoor',
                'state'       => 1,
                'page_id'     => $fixtures_hash->{partner_updated_indoor_page}{page_id},
            },
            {
                'update_time' => $UPDATE_TIME,
                'model'       => 'outdoor',
                'state'       => 1,
                'page_id'     => $fixtures_hash->{partner_updated_outdoor_page}{page_id},
            },
        );

        my @expected_blocks = (
            {
                'update_time' => $UPDATE_TIME,
                'model'       => 'indoor_block',
                'state'       => 1,
                %{$fixtures_hash->{partner_updated_indoor_video_block}}
            },
            {
                'update_time' => $UPDATE_TIME,
                'model'       => 'outdoor_block',
                'state'       => 1,
                %{$fixtures_hash->{partner_updated_outdoor_video_block}}
            },
        );

        cmp_bag(\@got_pages,  \@expected_pages,  'updated pages');
        cmp_bag(\@got_blocks, \@expected_blocks, 'updated blocks');

        is($app->kv_store->get($method->get_moderation_update_time_key('indoor')), $MOCKED_DATE, "indoor update time");
        is($app->kv_store->get($method->get_moderation_update_time_key('outdoor')), $MOCKED_DATE,
            "outdoor update time");

        @got_pages  = ();
        @got_blocks = ();

        $app->do('adv_net_dooh', 'send_updated_items_to_logbroker');

        cmp_bag(\@got_pages,  [], 'all pages are up to date');
        cmp_bag(\@got_blocks, [], 'all blocks are up to date');
    },
    application_package => 'Cron',
    fill_databases      => 0,
    fixtures            => \@fixtures,
    init                => [qw(api_selfservice)],
);
