#!/usr/bin/perl -w

use lib::abs;

use XML::Simple qw(XMLout);

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs restore_subs);
use Test::Partner::Utils qw(get_ports get_test_data_and_update_if_needed);

use Test::Differences qw(eq_or_diff);

use File::Temp qw(tempdir);
use File::Find;

use qbit;

BEGIN {
    no warnings 'redefine';
    *CORE::GLOBAL::time = sub() {1536152621};
}

$ENV{'SECRET_PARTNER2_CLICKHOUSE_PASSWORD'} = '';

my $DIRNAME;

local $$ = 100_500;

my $KEY_BY_NAME = {
    users_dict      => [1009],
    sellers_dict    => [345321935],
    pages_dict      => [41443],
    dsps_dict       => [1],
    monetizers_dict => [4],
    blocks_dict     => ["'context_on_site_rtb'", 142898, 1],
    video_categories_dict => [0,     128972],
    sources_dict          => [6],
    block_tags_dict       => [41443, 1],
};

run_tests(
    sub {
        my ($app) = @_;

        my @sql = ();
        mock_subs(
            {
                'QBit::Application::Model::DB::_do' => sub {
                    push(@sql, $_[1]);

                    return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::_do'}->(@_);
                },
                'QBit::Application::Model::DB::_get_all' => sub {
                    push(@sql, $_[1]);

                    return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::_get_all'}->(@_);
                },
            }
        );

        my $path = lib::abs::path('./update_dictionaries');
        foreach my $cron (qw(update_all_sources update_all_pages update_all_blocks)) {
            @sql = ();

            $app->do('dictionaries', $cron);

            my $sql = join("\n\n", @sql) . "\n";

            my $expected = get_test_data_and_update_if_needed("$cron.dump", $sql, raw => TRUE);

            eq_or_diff($sql, $expected, sprintf('sql correctly for "%s"', $cron));
        }

        init_clickhouse($app);

        check_dictionaries($app);
    },
    application_package => 'Cron',
    mocks               => ['mock_sort_sql'],
);

sub init_clickhouse {
    my ($app) = @_;

    my $dir = File::Temp->newdir(CLEANUP => 0);
    $DIRNAME = $dir->dirname();

    symlink '/etc/clickhouse-server/users.xml', "$DIRNAME/users.xml";

    my ($http_port, $tcp_port) = @{get_ports(count => 2)};

    my $config = sprintf(
        '<?xml version="1.0"?>
<yandex>
    <logger>
        <level>trace</level>
        <log>%s/clickhouse-server.log</log>
        <errorlog>%s/clickhouse-server.err.log</errorlog>
        <size>1000M</size>
        <count>10</count>
    </logger>
    <http_port>%d</http_port>
    <tcp_port>%d</tcp_port>
    <mark_cache_size>5368709120</mark_cache_size>
    <path>%s/</path>

    <users_config>users.xml</users_config>

    <default_profile>default</default_profile>
    <default_database>default</default_database>

    <dictionaries_config>*_dictionary.xml</dictionaries_config>

    <compression incl="clickhouse_compression">
    </compression>

    <format_schema_path>/var/lib/clickhouse/format_schemas/</format_schema_path>
</yandex>', $DIRNAME, $DIRNAME, $http_port, $tcp_port, $DIRNAME
    );

    writefile("$DIRNAME/config.xml", $config);

    my $dictionaries_dir = $app->get_option('ApplicationPath') . '/dictionaries';

    my @dictionaries = ();
    find(
        {
            wanted => sub {
                return if $File::Find::fullname !~ /\.json$/;

                my $content = readfile $File::Find::fullname;

                my $dictionary = from_json($content);

                push(@dictionaries, $dictionary);
            },
            follow => 1,
        },
        $dictionaries_dir
    );

    my $db_config = $app->get_option('partner_db');

    my $source = [
        {
            mysql => [
                {
                    port     => $db_config->{'port'},
                    user     => $db_config->{'user'},
                    password => $db_config->{'password'},
                    replica  => [
                        {
                            host     => $db_config->{'host'},
                            priority => 1,
                        }
                    ],
                    db    => $db_config->{'database'},
                    table => undef,
                }
            ]
        }
    ];

    foreach my $dict (@dictionaries) {
        my $table = $source->[0]{'mysql'}[0]{'table'} = $dict->{'source'}{'table'};

        my $dictionary_structure = {
            yandex => [
                {
                    dictionary => [
                        {
                            name      => $dict->{'name'},
                            source    => $source,
                            lifetime  => $dict->{'lifetime'},
                            layout    => {lc($dict->{'layout'}{'type'}) => ''},
                            structure => {
                                (
                                    exists($dict->{'structure'}{'id'})
                                    ? (id => {name => $dict->{'structure'}{'id'}{'name'}})
                                    : (
                                        key => {
                                            attribute => [
                                                map {{name => $_->{'name'}, type => $_->{'type'}}}
                                                  @{$dict->{'structure'}{'key'}{'attributes'}}
                                            ],
                                        }
                                      )
                                ),
                                attribute => [
                                    map {
                                        {name => $_->{'name'}, type => $_->{'type'}, null_value => $_->{'null_value'},}
                                      } @{$dict->{'structure'}{'attributes'}}
                                ],
                            },
                        }
                    ]
                }
            ]
        };

        XMLout(
            $dictionary_structure,
            RootName   => undef,
            NoAttr     => 1,
            OutputFile => sprintf('%s/%s_dictionary.xml', $DIRNAME, $table),
        );
    }

    `sudo chown clickhouse $DIRNAME -R`;

    my $code = system(
        sprintf(
'sudo -u clickhouse clickhouse-server --daemon --pid-file=%s/clickhouse-server.pid --config-file=%s/config.xml',
            $DIRNAME, $DIRNAME
        )
    );

    die "can't start clickhouse" if $code != 0;

    sleep(5);

    my $ch_config = $app->get_option('clickhouse_db');

    $ch_config->{'host'}          = 'localhost';
    $ch_config->{'port'}          = $http_port;
    $ch_config->{'balancer_port'} = $http_port;
    $ch_config->{'database'}      = 'default';
    $ch_config->{'user'}          = 'default';

    $app->set_option('clickhouse_db', $ch_config);

    $app->clickhouse_db->close_dbh();
    $app->clickhouse_db->_connect();

    restore_subs(['LWP::UserAgent::request']);
}

sub check_dictionaries {
    my ($app) = @_;

    my $dictionaries = $app->clickhouse_db->_get_all(
        q[SELECT
    name,
    key,
    attribute.names,
    attribute.types
FROM system.dictionaries FORMAT JSONCompact]
    );

    foreach my $dict (@$dictionaries) {
        throw "Set PK for dictionary $dict->{'name'}" unless $KEY_BY_NAME->{$dict->{'name'}};

        my $template_key;
        if ($dict->{'key'} =~ /^\((.+)\)$/) {
            my $types = $1;

            $template_key = 'tuple(' . join(', ', map {"to$_(%s)"} split(/,\s*/, $types)) . ')';
        } else {
            $template_key = "to$dict->{'key'}(%s)";
        }

        my $key = sprintf($template_key, @{$KEY_BY_NAME->{$dict->{'name'}}});

        my @fields = ();
        for (my $i = 0; $i < @{$dict->{'attribute.names'}}; $i++) {
            my $field_name = $dict->{'attribute.names'}[$i];
            my $type       = $dict->{'attribute.types'}[$i];

            push(@fields, "dictGet$type('$dict->{name}', '$field_name', $key) AS `$field_name`");
        }

        my $row = $app->clickhouse_db->_get_all('SELECT ' . join(', ', @fields) . ' FORMAT JSONCompact');

        my $expected = get_test_data_and_update_if_needed("data/$dict->{'name'}.json", $row);

        eq_or_diff($row, $expected, $dict->{'name'});
    }
}

sub kill_clickhouse {
    return unless defined($DIRNAME);

    my $code = system("sudo kill -9 `sudo cat $DIRNAME/clickhouse-server.pid`");

    die "can't kill clickhouse" if $code != 0;

    $code = system("sudo rm -rf $DIRNAME");

    die "can't delete folder" if $code != 0;
}

END {
    kill_clickhouse();
}
