#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Differences;
use Test::More tests => 2 + 11;

use Test::Partner2::Mock qw( mock_curdate );
use Test::Partner2::Simple;

my $mock_tm = '2018-09-13 18:00:00';
my $kv_tm   = '2018-09-13 17:30:00';    # -30 min
my $cron_tm = '2018-09-13 17:55:00';    # -5 min

my $app;
run_tests(
    sub {
        ($app) = @_;

        mock_curdate($mock_tm);

        no strict 'refs';
        no warnings 'redefine';

        local *{'QBit::Application::Model::KvStore::get'};
        local *{'QBit::Application::Model::KvStore::set'};
        local *{'Application::Model::SendMail::send'};

        *{'QBit::Application::Model::KvStore::get'} = sub {
            my ($self, $key) = @_;

            eq_or_diff($key, 'focus_domains_for_managers_mailing_lists', 'kv key name while get');

            return undef;    # incorrect value
        };

        *{'Application::Model::SendMail::send'} = sub {
            fail('validation: date is not specified');
        };

        *{'QBit::Application::Model::KvStore::set'} = sub {
            fail('validation: date is not specified');
        };

        try {
            $app->do(qw( domain check_new_owners ));
        }
        catch {
            my ($exception) = @_;

            isa_ok($exception, 'Exception::IncorrectParams',
                'validation: expected exception while date is not specified');
            eq_or_diff(
                $exception->message(),
                'From-date is not specified',
                'validation: expected exception message while date is not specified'
            );
        }
        finally {
            ok(shift, 'Exception throws');
        };

        *{'QBit::Application::Model::KvStore::get'} = sub {
            return $mock_tm;    # incorrect value (future)
        };

        *{'Application::Model::SendMail::send'} = sub {
            fail('validation: date is in the future');
        };

        *{'QBit::Application::Model::KvStore::set'} = sub {
            fail('validation: date is in the future');
        };

        try {
            $app->do(qw( domain check_new_owners ));
        }
        catch {
            my ($exception) = @_;

            isa_ok($exception, 'Exception::IncorrectParams',
                'validation: expected exception while date is in the future');

            eq_or_diff(
                $exception->message(),
                "From-date is after to-date",
                'validation: expected exception message while date is in the future'
            );
        }
        finally {
            ok(shift, 'Exception throws');
        };

        *{'QBit::Application::Model::KvStore::get'} = sub {
            return $kv_tm;    # correct value
        };
        *{'Application::Model::SendMail::send'} = sub {
            # First call (site)
            my ($self, %opts) = @_;

            eq_or_diff(
                \%opts,
                {
                    body => 'Domain "allrefrs.ru" (id: 5, tier: 2) '
                      . 'has new owner "mocked-yan-partner" (id: 1009) '
                      . 'since 2018-09-13 17:45:00',
                    content_type => 'text/plain',
                    from         => 'default',
                    subject      => 'tier: 2, domain: allrefrs.ru',
                    to           => 'login_tier_2@yandex-team.ru',
                },
                'SendMail params while normal call (site)'
            );

            *{'Application::Model::SendMail::send'} = sub {
                # Second call (video_an_site)
                my ($self, %opts) = @_;

                eq_or_diff(
                    \%opts,
                    {
                        body => 'Domain "kinopoisk.ru" (id: --, tier: 1) '
                          . 'has new owner "mocked-video-partner" (id: 1010) '
                          . 'since 2018-09-13 17:40:00',
                        content_type => 'text/plain',
                        from         => 'default',
                        subject      => 'tier: 1, domain: kinopoisk.ru',
                        to           => 'login_tier_1@yandex-team.ru',
                    },
                    'SendMail params while normal call (video_an_site)'
                );
            };
        };
        *{'QBit::Application::Model::KvStore::set'} = sub {
            my ($self, $key, $value) = @_;

            eq_or_diff($key, 'focus_domains_for_managers_mailing_lists', 'kv key name while set');
            eq_or_diff($value, $cron_tm, 'kv new value');
        };
        $app->do(qw( domain check_new_owners ));
    },
    application_package => 'Cron',
    db_suites           => {partner_db => [qw(focus_domains_for_managers_mailing_lists owner_site sites  users )],},
    init                => [qw(exception_dumper kv_store mailer)],
);
