#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Differences;
use Test::More tests => 2 + 9;

use Test::Partner2::Mock qw( mock_curdate );
use Test::Partner2::Simple;

use Exception::Validation::BadArguments;
use Exception::Validator::Errors;

my $tm = '2018-09-13 18:00:00';
my $id = 0;

my $app;
run_tests(
    sub {
        ($app) = @_;

        mock_curdate($tm);

        _check_values_in_db();

        subtest check_new => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';

            local *{'Cron::Methods::MailNotification::ERROR'} = sub {
                fail('unexpected error');
            };
            local *{'Cron::Methods::MailNotification::WARNF'} = sub {
                eq_or_diff(\@_, ['Following jobs were in empty state: %s', $id], 'expected warning');
            };
            local @ARGV = qw( mail_notification check check_new=1 try_prepare=0 try_send=0 );
            $app->do(qw( mail_notification ));

            _check_values_in_db(multistate => 1);
        };

        subtest try_prepare => sub {
            plan tests => 1;

            no strict 'refs';
            no warnings 'redefine';

            local *{'Cron::Methods::MailNotification::ERROR'} = sub {
                fail('unexpected error');
            };
            local @ARGV = qw( mail_notification check check_new=0 try_prepare=1 try_send=0 );
            $app->do(qw( mail_notification ));

            _check_values_in_db(multistate => 2);
        };

        subtest try_send => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';

            local *{'Cron::Methods::MailNotification::ERROR'} = sub {
                fail('unexpected error');
            };
            local *{'Cron::Methods::MailNotification::INFOF'} = sub {
                fail('unexpected error');
            };
            local *{'Application::Model::SendMail::send'} = sub {
                my ($self, %opts) = @_;

                eq_or_diff(
                    \%opts,
                    {
                        body => {
                            content_type => 'text/html',
                            type         => 'TT2',
                            template     => 'mail_notification/base.html.tt2',
                            vars         => {
                                message_body       => '',
                                plain_text_wrapper => '',
                            },
                        },
                        content_type => 'text/html',
                        from         => 'default',
                        subject      => gettext('Yandex Advertising Network'),
                        to           => {user_id => '1009'},
                    },
                    'SendMail options'
                );
            };
            local @ARGV = qw( mail_notification check check_new=0 try_prepare=0 try_send=1 );
            $app->do(qw( mail_notification ));

            _check_values_in_db(multistate => 4);
        };

        subtest do_periodicity => sub {
            plan tests => 1;

            $app->mail_notification->partner_db_table()->edit(0, {multistate => 2, period => 300,});
            $app->mail_notification->do_action(0, 'periodicity');

            _check_values_in_db(multistate => 18, period => 300,);
        };

        subtest try_to_send_while_timeout => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';

            local *{'Cron::Methods::MailNotification::ERROR'} = sub {
                fail('unexpected error');
            };
            local *{'Cron::Methods::MailNotification::INFOF'} = sub {
                eq_or_diff(\@_, ['Task %s postponed due to timeout', 0], 'info message about timeout');
            };
            local *{'Application::Model::SendMail::send'} = sub {
                fail('unexpected error');
            };
            local @ARGV = qw( mail_notification check check_new=0 try_prepare=0 try_send=1 );
            $app->do(qw( mail_notification ));

            _check_values_in_db(multistate => 18, period => 300,);
        };

        subtest try_send_periodicity => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';

            local *{'Application::Model::MailNotification::_check_timeout'} = sub {
                return TRUE;
            };
            local *{'Cron::Methods::MailNotification::ERROR'} = sub {
                fail('unexpected error');
            };
            local *{'Cron::Methods::MailNotification::INFOF'} = sub {
                fail('unexpected error');
            };
            local *{'Application::Model::SendMail::send'} = sub {
                my ($self, %opts) = @_;

                eq_or_diff(
                    \%opts,
                    {
                        body => {
                            content_type => 'text/html',
                            type         => 'TT2',
                            template     => 'mail_notification/base.html.tt2',
                            vars         => {
                                message_body       => '',
                                plain_text_wrapper => '',
                            },
                        },
                        content_type => 'text/html',
                        from         => 'default',
                        subject      => gettext('Yandex Advertising Network'),
                        to           => {user_id => '1009'},
                    },
                    'SendMail options'
                );
            };
            local @ARGV = qw( mail_notification check check_new=0 try_prepare=0 try_send=1 );
            $app->do(qw( mail_notification ));

            _check_values_in_db(date_x => '2018-09-13 18:05:00', multistate => 20, period => 300,);
        };

        subtest try_prepare_periodicity => sub {
            plan tests => 1;

            mock_curdate('2018-09-13 18:05:01');

            no strict 'refs';
            no warnings 'redefine';

            local *{'Cron::Methods::MailNotification::ERROR'} = sub {
                fail('unexpected error');
            };
            local @ARGV = qw( mail_notification check check_new=0 try_prepare=1 try_send=0 );
            $app->do(qw( mail_notification ));

            _check_values_in_db(date_x => '2018-09-13 18:05:00', multistate => 18, period => 300,);
        };

        subtest try_prepare_blocked_user => sub {
            plan tests => 1;

            $app->mail_notification->partner_db_table()->edit(
                0,
                {
                    multistate => 1,
                    period     => undef,
                    user_id    => 0,       # у крон-юзера нет ролей = заблокирован
                }
            );

            no strict 'refs';
            no warnings 'redefine';

            local *{'Cron::Methods::MailNotification::ERROR'} = sub {
                fail('unexpected error');
            };
            local @ARGV = qw( mail_notification check check_new=0 try_prepare=1 try_send=0 );
            $app->do(qw( mail_notification ));

            _check_values_in_db(date_x => '2018-09-13 18:05:00', multistate => 10, user_id => 0,);
        };
    },
    application_package => 'Cron',
    db_suites           => {partner_db => [qw( users mail_notification)],},
    init                => [qw(mail_notification mailer)]
);

sub _check_values_in_db {
    eq_or_diff(
        $app->mail_notification->partner_db_table()->get_all(
            fields => [qw(id multistate type user_id period create_date date_x opts)],
            filter => {id => $id},
        ),
        [
            {
                create_date => '2018-09-13 16:30:00',
                date_x      => '2018-09-13 17:00:00',
                id          => $id,
                multistate  => 0,
                opts        => '{}',
                period      => undef,
                type        => 0,
                user_id     => 1009,
                @_,
            },
        ],
        'check values in db'
    );
}
