#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Differences;
use Test::More tests => 44;

use Test::Partner2::Mock qw(mock_curdate);
use Test::Partner2::Simple;

use PiConstants qw($MYSQL_MIN_DATETIME);

my $mocked_date = '2018-07-03 12:00:00';    # 1530619200

my $get_all_mobile_app_request = {
    'fields' => [qw(id create_date type store_id store_url owners)],
    'filter' => ['AND', [['multistate', '=', 'need_approve'], ['waiting_moderation', '=', $MYSQL_MIN_DATETIME],],],
};
my $get_all_site_request = {
    'fields' => [qw(id domain create_date data_for_site_stat)],
    'filter' => ['AND', [['multistate', '=', 'need_approve'], ['waiting_moderation', '=', $MYSQL_MIN_DATETIME],],],
};
my %mirror_query = (
    context_on_site_mirrors => q[SELECT
    `context_on_site_mirrors`.`campaign_id` AS `campaign_id`,
    `users`.`client_id` AS `client_id`,
    `context_on_site_mirrors`.`create_date` AS `create_date`,
    `site`.`domain` AS `domain`,
    `context_on_site_mirrors`.`id` AS `id`,
    `context_on_site_mirrors`.`domain` AS `mirror`,
    `context_on_site_campaign`.`owner_id` AS `owner_id`
FROM `context_on_site_mirrors`
INNER JOIN `context_on_site_campaign` ON (
    `context_on_site_campaign`.`id` = `context_on_site_mirrors`.`campaign_id`
)
INNER JOIN `site` ON (
    `site`.`id` = `context_on_site_campaign`.`domain_id`
)
INNER JOIN `users` ON (
    `users`.`id` = `context_on_site_campaign`.`owner_id`
)
WHERE (
    `context_on_site_mirrors`.`multistate` IN ('0')
    AND `context_on_site_mirrors`.`waiting_moderation` = '1000-01-01 00:00:00'
)],
    search_on_site_mirrors => q[SELECT
    `search_on_site_mirrors`.`campaign_id` AS `campaign_id`,
    `users`.`client_id` AS `client_id`,
    `search_on_site_mirrors`.`create_date` AS `create_date`,
    `site`.`domain` AS `domain`,
    `search_on_site_mirrors`.`id` AS `id`,
    `search_on_site_mirrors`.`domain` AS `mirror`,
    `search_on_site_campaign`.`owner_id` AS `owner_id`
FROM `search_on_site_mirrors`
INNER JOIN `search_on_site_campaign` ON (
    `search_on_site_campaign`.`id` = `search_on_site_mirrors`.`campaign_id`
)
INNER JOIN `site` ON (
    `site`.`id` = `search_on_site_campaign`.`domain_id`
)
INNER JOIN `users` ON (
    `users`.`id` = `search_on_site_campaign`.`owner_id`
)
WHERE (
    `search_on_site_mirrors`.`multistate` IN ('0')
    AND `search_on_site_mirrors`.`waiting_moderation` = '1000-01-01 00:00:00'
)],
);

my $mocked_indoor_block = {
    page_id         => 111,
    id              => 1,
    public_id       => 'I-D-111-1',
    caption         => 'indoor block caption',
    address         => 'indoor block address',
    gps             => '12,12',
    photo_id_list   => [1, 2],
    photo_link_list => [
        {
            "id"    => "1",
            "links" => {
                "orig" => {
                    "height" => 1200,
                    "path"   => "/get-partner/2154589/2a0000016b46c054ea5ea4f730ffdbcb4513/orig",
                    "width"  => 1600
                }
            },
            "md5" => "b0720bce49deff162eb6a00ee5b4e7e1"
        },
        {
            "id"    => "2",
            "links" => {
                "orig" => {
                    "height" => 853,
                    "path"   => "/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3fe/orig",
                    "width"  => 1280
                }
            },
            "md5" => "912e231b0c3d100479d28ef17eb7f018"
        }
    ],
    login           => 'indoor_block_owner_login',
    owner_id        => 111666,
    owner_client_id => 666111,
    moderation      => {
        photo_id_list => {
            1 => {verdict => 0, request_id => ''},
            2 => {verdict => 0, request_id => ''},
        },
    }
};

$mocked_indoor_block->{opts} = {
    moderation    => $mocked_indoor_block->{moderation},
    photo_id_list => $mocked_indoor_block->{photo_id_list},
};

my $mocked_outdoor_block = {
    page_id         => 112,
    id              => 1,
    public_id       => 'O-D-112-1',
    caption         => 'outdoor block caption',
    address         => 'outdoor block address',
    gps             => '11,66',
    photo_id_list   => [3, 4],
    photo_link_list => [
        {
            "id"    => "3",
            "links" => {
                "orig" => {
                    "height" => 1200,
                    "path"   => "/get-partner/2154589/2a0000016b46c054ea5ea4f730ffdbcb4513/orig",
                    "width"  => 1600
                }
            },
            "md5" => "b0720bce49deff162eb6a00ee5b4e7e1"
        },
        {
            "id"    => "4",
            "links" => {
                "orig" => {
                    "height" => 853,
                    "path"   => "/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3fe/orig",
                    "width"  => 1280
                }
            },
            "md5" => "912e231b0c3d100479d28ef17eb7f018"
        }
    ],
    login           => 'outdoor_block_owner_login',
    owner_id        => 666111,
    owner_client_id => 111666,
    moderation      => {
        address       => {'outdoor block address' => {verdict => 0, request_id => ''},},
        caption       => {'outdoor block caption' => {verdict => 1, request_id => '121'},},
        gps           => {'11,66'                 => {verdict => 0, request_id => ''},},
        photo_id_list => {
            3 => {verdict => 0, request_id => '123'},
            4 => {verdict => 0, request_id => '124'},
        },
    }
};

$mocked_outdoor_block->{opts} = {
    moderation    => $mocked_outdoor_block->{moderation},
    photo_id_list => $mocked_outdoor_block->{photo_id_list},
};

my $mocked_indoor_page = {
    page_id    => 111,
    id         => 1,
    caption    => 'indoor page caption',
    login      => 'indoor_page_owner_login',
    owner_id   => 111666,
    client_id  => 666111,
    moderation => {
        caption => {'indoor page caption' => {verdict => -1, request_id => '123'},},
        address => {'indoor page address' => {}},
        gps     => {'22,77'               => {}},
    },
    address => 'indoor page address',
    gps     => '22,77',
};
$mocked_indoor_page->{opts} = {moderation => $mocked_indoor_page->{moderation}};

my $mocked_outdoor_page = {
    page_id    => 111,
    id         => 1,
    caption    => 'outdoor page caption',
    login      => 'outdoor_page_owner_login',
    owner_id   => 111666,
    client_id  => 666111,
    moderation => {caption => {'outdoor page caption' => {},},},
};
$mocked_outdoor_page->{opts} = {moderation => $mocked_outdoor_page->{moderation}};

my %data = (
    context_on_site_mirrors => [
        {
            data => {
                client_id   => '20306495',
                create_date => '2018-07-02 23:54:35',
                domain      => 'jewellerymag.ru',
                mirror      => 'm.jewellerymag.ru',
                yuid        => '34705945',
            },
            meta    => {id => 63153, model => 'context_on_site_mirrors', request_id => '1234567890123456'},
            service => 'pi',
            type    => 'mirror',
        },
    ],
    mobile_app => [
        {
            data => {
                client_id   => '20306495',
                create_date => '2018-07-02 23:54:35',
                store_id    => 'com.yandex',
                store_url   => 'https://yandex.ru',
                type        => 1,
                yuid        => '34705945',
            },
            meta    => {id => 63153, model => 'mobile_app', request_id => '1234567890123456'},
            service => 'pi',
            type    => 'bundle',
        },
    ],
    search_on_site_mirrors => [
        {
            data => {
                client_id   => '20306495',
                create_date => '2018-07-02 23:54:35',
                domain      => 'jewellerymag.ru',
                mirror      => 'm.jewellerymag.ru',
                yuid        => '34705945',
            },
            meta    => {id => 63153, model => 'search_on_site_mirrors', request_id => '1234567890123456'},
            service => 'pi',
            type    => 'mirror',
        },
    ],
    site => [
        {
            data => {
                client_id   => '20306495',
                create_date => '2018-07-02 23:54:35',
                domain      => 'jewellerymag.ru',
                yuid        => '34705945',
            },
            meta    => {id => 63153, model => 'site', request_id => '1234567890123456'},
            service => 'pi',
            type    => 'site',
        }
    ],
    indoor_block => [
        {
            data => {
                image_hash => 'b0720bce49deff162eb6a00ee5b4e7e1',
                image_url =>
                  'https://avatars.mdst.yandex.net/get-partner/2154589/2a0000016b46c054ea5ea4f730ffdbcb4513/orig'
            },
            meta => {
                client_id  => 666111,
                imp_id     => 1,
                model      => 'indoor_block',
                page_id    => 111,
                user_id    => 111666,
                request_id => '1234567890123456'
            },
            service => 'pi',
            type    => 'indoor_block_image'
        },
        {
            data => {
                image_hash => '912e231b0c3d100479d28ef17eb7f018',
                image_url =>
                  'https://avatars.mdst.yandex.net/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3fe/orig'
            },
            meta => {
                client_id  => 666111,
                imp_id     => 1,
                model      => 'indoor_block',
                page_id    => 111,
                user_id    => 111666,
                request_id => '1234567890123457'
            },
            service => 'pi',
            type    => 'indoor_block_image'
        },
    ],
    outdoor_block => [
        {
            data => {
                'address' => 'outdoor block address',
                'gps'     => '11,66'
            },
            meta => {
                'client_id' => 111666,
                'imp_id'    => 1,
                'model'     => 'outdoor_block',
                'page_id'   => 112,
                'user_id'   => 666111,
                request_id  => '1234567890123456'
            },
            service => 'pi',
            type    => 'outdoor_block_address'
        },
    ],
    indoor => [
        {
            data => {
                address => 'indoor page address',
                gps     => '22,77'
            },
            meta => {
                client_id  => 666111,
                model      => 'indoor',
                page_id    => 111,
                user_id    => 111666,
                request_id => '1234567890123456',
            },
            service => 'pi',
            type    => 'indoor_page_address',
        }
    ],
    outdoor => [
        {
            data => {caption => 'outdoor page caption',},
            meta => {
                client_id  => 666111,
                model      => 'outdoor',
                page_id    => 111,
                user_id    => 111666,
                request_id => '1234567890123456',
            },
            service => 'pi',
            type    => 'outdoor_page_caption',
        }
    ],
);

my $dooh_edit = {
    indoor_block => {
        moderation => {
            photo_id_list => {
                1 => {
                    request_id => 1,
                    verdict    => 0,
                },
                2 => {
                    request_id => 2,
                    verdict    => 0,
                },
            },
        },
        waiting_moderation => $mocked_date,
    },
    outdoor_block => {
        moderation => {
            address => {
                'outdoor block address' => {
                    request_id => 1,
                    verdict    => 0,
                },
            },
            caption => {
                'outdoor block caption' => {
                    request_id => '121',
                    verdict    => 1,
                },
            },
            gps => {
                '11,66' => {
                    request_id => 1,
                    verdict    => 0,
                },
            },
            photo_id_list => {
                3 => {
                    request_id => '123',
                    verdict    => 0,
                },
                4 => {
                    request_id => '124',
                    verdict    => 0,
                },
            },
        },
        waiting_moderation => $mocked_date,
    },
    indoor => {
        moderation => {
            address => {
                'indoor page address' => {
                    request_id => 1,
                    verdict    => 0
                }
            },
            caption => {
                'indoor page caption' => {
                    request_id => '123',
                    verdict    => -1
                }
            },
            gps => {
                '22,77' => {
                    request_id => 1,
                    verdict    => 0
                }
            }
        },
        waiting_moderation => $mocked_date
    },
    outdoor => {
        moderation => {
            caption => {
                'outdoor page caption' => {
                    request_id => 1,
                    verdict    => 0
                }
            },
        },
        waiting_moderation => $mocked_date
    },
};

my @model =
  sort (qw(context_on_site_mirrors mobile_app search_on_site_mirrors site indoor_block outdoor_block indoor outdoor));

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate($mocked_date);

        no strict 'refs';
        no warnings 'redefine';

        my $get_all = \&QBit::Application::Model::DB::Query::get_all;
        local *{'Application::Model::Product::AN::Site::get_all'};
        local *{'QBit::Application::Model::DB::Query::get_all'};
        local *{'Cron::Methods::Moderation::INFO'};
        local *{'Application::Model::API::Yandex::SelfService::logbroker'} = sub { };
        local *{'QBit::Application::Model::DB::mysql::Table::edit'};
        local *{'Time::HiRes::time'} = sub {
            return '1234567890.123456';
        };

        my $n;
        *{'Application::Model::Product::AN::MobileApp::get_all'} = sub {
            my ($self, %opts) = @_;
            eq_or_diff(\%opts, $get_all_mobile_app_request, 'check_need_approve: get_all mobile_app params (empty)');
            return [];
        };
        *{'Application::Model::Product::AN::Site::get_all'} = sub {
            my ($self, %opts) = @_;
            eq_or_diff(\%opts, $get_all_site_request, 'check_need_approve: get_all site params (empty)');
            return [];
        };
        *{'QBit::Application::Model::DB::Query::get_all'} = sub {
            if (caller eq 'Cron::Methods::Moderation') {
                my ($self, %opts) = @_;

                eq_or_diff(\%opts, {}, "check_need_approve: get_all query params $model[$n] (empty)");
                eq_or_diff(
                    [$self->get_sql_with_data()],
                    [$mirror_query{$model[$n]}],
                    "check_need_approve: get_all query $model[$n] (empty)"
                );

                return [];
            }
            goto &$get_all;
        };
        *{'Cron::Methods::Moderation::INFOF'} = sub {
            eq_or_diff(
                \@_,
                ['There is no elements in %s to proceed', $model[$n]],
                "check_need_approve: info $model[$n] (empty)"
            );
            $n++;    # This is the last method in iteration
        };
        *{'Application::Model::API::Yandex::SelfService::logbroker'} = sub {
            fail('check_need_approve: logbroker (empty)');
        };
        *{'QBit::Application::Model::DB::mysql::Table::edit'} = sub {
            fail('check_need_approve: table edit (empty)');
        };
        $n = 0;
        $app->do(qw( moderation check_need_approve ));

        *{'Application::Model::Product::AN::MobileApp::get_all'} = sub {
            return [
                {
                    create_date => '2018-07-02 23:54:35',
                    id          => '63153',
                    owners      => [
                        {
                            client_id => '20306495',
                            id        => '34705945',
                            full_name => 'Прозвицкая Мария Баходировна',
                            login     => 'aksi2004',
                        },
                    ],
                    store_id  => 'com.yandex',
                    store_url => 'https://yandex.ru',
                    type      => 1,
                }
            ];
        };
        *{'Application::Model::Product::AN::Site::get_all'} = sub {
            return [
                {
                    data_for_site_stat => [
                        {
                            owner => {
                                client_id => '20306495',
                                id        => '34705945',
                                full_name => 'Прозвицкая Мария Баходировна',
                                login     => 'aksi2004',
                            },
                            domain_id => '63153',
                            user_id   => '34705945',
                        },
                    ],
                    domain      => 'jewellerymag.ru',
                    create_date => '2018-07-02 23:54:35',
                    id          => '63153',
                }
            ];
        };
        *{'Application::Model::Block::InDoor::get_all'} = sub {
            return [$mocked_indoor_block];
        };
        *{'Application::Model::Block::OutDoor::get_all'} = sub {
            return [$mocked_outdoor_block];
        };
        *{'Application::Model::Page::InDoor::get_all'} = sub {
            return [$mocked_indoor_page];
        };
        *{'Application::Model::Page::OutDoor::get_all'} = sub {
            return [$mocked_outdoor_page];
        };
        *{'QBit::Application::Model::DB::Query::get_all'} = sub {
            if (caller eq 'Cron::Methods::Moderation') {
                return [
                    {
                        campaign_id => '123456',
                        client_id   => '20306495',
                        create_date => '2018-07-02 23:54:35',
                        domain      => 'jewellerymag.ru',
                        id          => '63153',
                        mirror      => 'm.jewellerymag.ru',
                        owner_id    => '34705945',
                    }
                ];
            }
            goto &$get_all;
        };
        *{'Cron::Methods::Moderation::INFOF'} = sub {
            eq_or_diff(
                \@_,
                ['Received %d elements in %s to proceed', 1, $model[$n]],
                "check_need_approve: info $model[$n] (1 site)"
            );
        };
        *{'Application::Model::API::Yandex::SelfService::logbroker'} = sub {
            my ($self, %opts) = @_;
            my ($file, $data) = @opts{qw(topic data)};
            eq_or_diff($data, $data{$model[$n]}, 'check_need_approve: logbroker data (1 ' . $model[$n] . ')');
            for (my $i = 0; $i < @$data; $i++) {
                $data->[$i]{meta}{request_id} = $i + 1;
            }
        };

        *{'Application::Model::Block::Dooh::do_action'} = sub {
            my ($self, $obj, $action, %data) = @_;

            eq_or_diff($action, 'edit', 'dooh block do_action edit');
            eq_or_diff(\%data, $dooh_edit->{$model[$n]}, "dooh block do_action edit patch $model[$n]");
            $n++;    # This is the last method in iteration
        };
        *{'Application::Model::Page::Dooh::do_action'} = sub {
            my ($self, $obj, $action, %data) = @_;

            eq_or_diff($action, 'edit', 'dooh page do_action edit');
            eq_or_diff(\%data, $dooh_edit->{$model[$n]}, "dooh page do_action edit patch $model[$n]");
            $n++;    # This is the last method in iteration
        };
        *{'QBit::Application::Model::DB::mysql::Table::edit'} = sub {
            my ($self, @data) = @_;

            my $meta =
              (ref($data{$model[$n]}) eq 'ARRAY')
              ? $data{$model[$n]}->[0]->{meta}
              : $data{$model[$n]}->{meta};

            my $pk_list =
              exists $meta->{id}
              ? [[$meta->{id}]]
              : [[$meta->{page_id}, $meta->{imp_id}]];

            if (ref $data[0] eq 'HASH') {
                $data[0] = [[@{$data[0]}{qw(page_id id)}]];
            }

            eq_or_diff(
                \@data,
                [$pk_list, {'waiting_moderation' => $mocked_date}],
                "check_need_approve: query ($n) '$model[$n]'"
            );
            $n++;    # This is the last method in iteration
        };
        $n = 0;
        $app->do(qw( moderation check_need_approve ));

    },
    'application_package' => 'Cron',
    'do_not_die_on_fail'  => TRUE,
    'init'                => [qw(api_selfservice)],
);
