#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::More;
use Test::Exception;

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs restore_subs mock_curdate);
use qbit;

my $HOST = 'mocked_host.org';
mock_curdate("2019-10-31 17:11:07");

my $cert_str_ok_date = q(notAfter=Oct 22 16:12:03 2020 GMT
    -----BEGIN CERTIFICATE-----
        MIIFGTCCBAGgAwIBAgITFwAASePAapg473YF+AAAAABJ4zANBgkqhkiG9w0BAQsF
            QCt5Dp28LwplvbGVrQ==
                -----END CERTIFICATE-----
    );
my %CRIT = (
    events => [
        {
            service => 'yamoney-ssl',
            status  => 'CRIT',
            host    => $HOST,
        }
    ]
);
my %OK = (
    events => [
        {
            service => 'yamoney-ssl',
            status  => 'OK',
            host    => $HOST,
        }
    ]
);
my %WARN = (
    events => [
        {
            service => 'yamoney-ssl',
            status  => 'WARN',
            host    => $HOST,
        }
    ]
);

run_tests(
    sub {
        my ($app) = @_;

        $app->set_option('hostname', $HOST);

        subtest ok_sent_on_gt => sub {
            plan tests => 1;
            mock_subs(
                {
                    'Partner2::Juggler::API::send' => sub {
                        my ($self, %opts) = @_;
                        is_deeply(\%opts, \%OK, 'ok sent: 28 <= 30');
                    },
                    'Cron::Methods::System::_get_cert_expires_in_days' => sub {
                        return 30;
                    },
                }
            );
            $app->do('system', 'check_yamoney_cert_expiration');
        };

        subtest warn_sent_on_eq => sub {
            plan tests => 1;
            restore_subs(['Cron::Methods::System::_get_cert_expires_in_days', 'Partner2::Juggler::API::send']);
            mock_subs(
                {
                    'Partner2::Juggler::API::send' => sub {
                        my ($self, %opts) = @_;
                        is_deeply(\%opts, \%WARN, 'WARN sent: 14 <= 27');
                    },
                    'Cron::Methods::System::_get_cert_expires_in_days' => sub {
                        return 27;
                    },
                }
            );
            $app->do('system', 'check_yamoney_cert_expiration');
        };

        subtest crit_sent_on_zero => sub {
            plan tests => 1;
            restore_subs(['Cron::Methods::System::_get_cert_expires_in_days', 'Partner2::Juggler::API::send']);
            mock_subs(
                {
                    'Partner2::Juggler::API::send' => sub {
                        my ($self, %opts) = @_;
                        is_deeply(\%opts, \%CRIT, 'CRIT sent: 28 >= 0');
                    },
                    'Cron::Methods::System::_get_cert_expires_in_days' => sub {
                        return 0;
                    },
                }
            );
            $app->do('system', 'check_yamoney_cert_expiration');
        };

        subtest crit_sent_on_negative => sub {
            plan tests => 1;
            restore_subs(['Cron::Methods::System::_get_cert_expires_in_days', 'Partner2::Juggler::API::send']);
            mock_subs(
                {
                    'Partner2::Juggler::API::send' => sub {
                        my ($self, %opts) = @_;
                        is_deeply(\%opts, \%CRIT, 'CRIT sent: 28 >= -1');
                    },
                    'Cron::Methods::System::_get_cert_expires_in_days' => sub {
                        return -1;
                    },
                }
            );
            $app->do('system', 'check_yamoney_cert_expiration');
        };

        subtest parse_cert_expire_date => sub {

            my $got = Cron::Methods::System::_parse_cert_expire_date($cert_str_ok_date);
            is_deeply($got, [2020, 10, 22], 'date parsed - ok');

            my $cert_str_notok = q(notAfter=10 22 16:12:03 2020 GMT
-----BEGIN CERTIFICATE-----
MIIFGTCCBAGgAwIBAgITFwAASePAapg473YF+AAAAABJ4zANBgkqhkiG9w0BAQsF
QCt5Dp28LwplvbGVrQ==
-----END CERTIFICATE-----
);
            throws_ok(
                sub {
                    Cron::Methods::System::_parse_cert_expire_date($cert_str_notok);
                },
                'Exception',
                'date not parsed - exception thrown'
            );
        };

        restore_subs(
            ['Cron::Methods::System::_get_cert_expires_in_days', 'Cron::Methods::System::_parse_cert_expire_date']);
        mock_subs(
            {
                'Cron::Methods::System::_get_cert_expire_date_from_file' => sub {
                    return q(notAfter=Oct 22 16:12:03 2020 GMT
-----BEGIN CERTIFICATE-----
MIIFGTCCBAGgAwIBAgITFwAASePAapg473YF+AAAAABJ4zANBgkqhkiG9w0BAQsF
QCt5Dp28LwplvbGVrQ==
-----END CERTIFICATE-----
);
                },
            }
        );
        my $got = Cron::Methods::System::_get_cert_expires_in_days('filepath');
        is($got, 357, 'get_cert_expires_in_days positive - ok');

        restore_subs(['Cron::Methods::System::_get_cert_expire_date_from_file']);
        mock_subs(
            {
                'Cron::Methods::System::_get_cert_expire_date_from_file' => sub {
                    return q(notAfter=Oct 30 16:12:03 2019 GMT
-----BEGIN CERTIFICATE-----
MIIFGTCCBAGgAwIBAgITFwAASePAapg473YF+AAAAABJ4zANBgkqhkiG9w0BAQsF
QCt5Dp28LwplvbGVrQ==
-----END CERTIFICATE-----
);
                },
            }
        );
        $got = Cron::Methods::System::_get_cert_expires_in_days('filepath');
        is($got, -1, 'get_cert_expires_in_days negative - ok');
    },
    application_package  => 'Cron',
    dont_create_database => TRUE,
    mocks                => {mock_juggler => FALSE},
);
