#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use File::Temp qw(tempfile);

use Partner2::Juggler::API;

use Test::Partner2::Simple;
use Test::More;
use Test::Differences qw(eq_or_diff);

use qbit;

my $FIXTURES = [
    {
        name            => 'without pkg_version and config',
        pkg_version     => undef,
        config          => undef,
        error           => 'Expected version',
        send_to_juggler => FALSE,
        check_version   => undef,
    },
    {
        name            => 'without config',
        pkg_version     => '0.01',
        config          => undef,
        expected        => ['checks_url', 'host', 'juggler_token', 'md5_hex_config', 'namespace', 'push_url',],
        send_to_juggler => TRUE,
        check_version   => '0.01_99914b932bd37a50b983c5e7c90ae93b',
    },
    {
        name            => 'without pkg_version',
        pkg_version     => undef,
        config          => '{}',
        error           => 'Expected version',
        send_to_juggler => FALSE,
        check_version   => '0.01_99914b932bd37a50b983c5e7c90ae93b',
    },
    {
        name        => 'pkg_version 0.01',
        pkg_version => '0.01',
        config      => '{"logins":["LOGIN"]}',
        expected    => ['checks_url', 'host', 'juggler_token', 'logins', 'md5_hex_config', 'namespace', 'push_url',],
        send_to_juggler => TRUE,
        check_version   => '0.01_3c3a2b11a40a77e63d198b7ce4452eff',
    },
    {
        name            => 'without change',
        pkg_version     => '0.01',
        config          => '{"logins":["LOGIN"]}',
        expected        => TRUE,
        send_to_juggler => FALSE,
        check_version   => '0.01_3c3a2b11a40a77e63d198b7ce4452eff',
    },
    {
        name        => 'pkg_version 0.02',
        pkg_version => '0.02',
        config      => '{"logins":["LOGIN"]}',
        expected    => ['checks_url', 'host', 'juggler_token', 'logins', 'md5_hex_config', 'namespace', 'push_url',],
        send_to_juggler => TRUE,
        check_version   => '0.02_3c3a2b11a40a77e63d198b7ce4452eff',
    },
    {
        name            => 'pkg_version 0.01 again',
        pkg_version     => '0.01',
        config          => '{"logins":["LOGIN"]}',
        send_to_juggler => TRUE,
        expected => ['checks_url', 'host', 'juggler_token', 'logins', 'md5_hex_config', 'namespace', 'push_url',],
        send_to_juggler => TRUE,
        check_version   => '0.01_3c3a2b11a40a77e63d198b7ce4452eff',
    },
    {
        name            => 'changed config',
        pkg_version     => '0.01',
        config          => '{"logins":["LOGIN","LOGIN2"]}',
        send_to_juggler => TRUE,
        expected => ['checks_url', 'host', 'juggler_token', 'logins', 'md5_hex_config', 'namespace', 'push_url',],
        send_to_juggler => TRUE,
        check_version   => '0.01_4b38045b42a20219d7cd47d4b7586203',
    },
];

run_tests(
    sub {
        my ($app) = @_;

        my ($NAME, $EXPECTED, $SEND_TO_JUGGLER);

        {
            no strict 'refs';
            no warnings 'redefine';

            *{'Cron::generate_juggler'} = sub {
                my ($self, $pja) = @_;

                $SEND_TO_JUGGLER = TRUE;

                eq_or_diff([sort keys(%{$pja->{'__CONFIG__'}})],
                    $EXPECTED, $NAME . ': config ok in Cron::generate_juggler');

                return TRUE;
            };

            *{'Application::Model::Statistics::generate_juggler'} = sub {
                my ($self, $pja) = @_;

                $SEND_TO_JUGGLER = TRUE;

                eq_or_diff([sort keys(%{$pja->{'__CONFIG__'}})],
                    $EXPECTED, $NAME . ': config ok in Application::Model::Statistics::generate_juggler');

                return TRUE;
            };
        }

        foreach my $test (@$FIXTURES) {
            my ($name, $pkg_version, $config, $error, $expected, $send_to_juggler, $check_version) =
              @$test{qw(name pkg_version config error expected send_to_juggler check_version)};

            $NAME            = $name;
            $EXPECTED        = $expected;
            $SEND_TO_JUGGLER = FALSE;

            if (defined($pkg_version)) {
                $app->set_option('version' => $pkg_version);
            }

            my $filename = '';
            if (defined($config)) {
                $filename = "/tmp/config$$.json";

                writefile($filename, $config);
            }

            $Partner2::Juggler::API::CONFIG_PATH = $filename;

            my $got_error;
            try {
                $app->do('system', 'create_juggler_checks');
            }
            catch {
                $got_error = shift->message;
            };

            if ($got_error || $error) {
                is($got_error, $error, $NAME . ': check error');
            }

            is($SEND_TO_JUGGLER, $send_to_juggler, $NAME . ': check send to juggler');

            is($app->kv_store->get('juggler_checks_version'), $check_version, $NAME . ': check version');

            #reset
            $app->set_option('version' => undef);
            unlink $filename if $filename;
            $SEND_TO_JUGGLER = FALSE;
        }
    },
    application_package => 'Cron',
    init                => [qw(statistics)],
);
