#!/usr/bin/perl

use Test::More tests => 3;
use Test::Differences qw(eq_or_diff);

use qbit;

use QBit::Validator;
use Test::Partner2::Simple;
use Test::Partner::Utils qw(get_test_data_and_update_if_needed);

my @STAGES = qw(preprod production test autotest);

# Сборка регэкспа для маски крона
my $min_rx     = qr/(?:[1-5]?\d)/;
my $min_set_rx = qr/(?:\*|(?:$min_rx-)?$min_rx)(?:\/$min_rx)?/;

my $hour_rx     = qr/(?:2[0-3]|1\d|\d)/;
my $hour_set_rx = qr/(?:\*|(?:$hour_rx-)?$hour_rx)(?:\/$hour_rx)?/;

my $day_rx     = qr/(?:3[01]|[12]\d|\d)/;
my $day_set_rx = qr/(?:\*|(?:$day_rx-)?$day_rx)(?:\/$day_rx)?/;

my $month_rx     = qr/(?:1[0-2]|\d)/;
my $month_set_rx = qr/(?:\*|(?:$month_rx-)?$month_rx)(?:\/$month_rx)?/;

my $week_rx     = qr/(?:[0-6])/;
my $week_set_rx = qr/(?:\*|(?:$week_rx-)?$week_rx)(?:\/$week_rx)?/;

# Без учёта макросов: @yearly, @annually, @monthly, @weekly, @daily, @hourly, @reboot, @every
my $cron_rx = qr/^\s*
    (?:$min_set_rx,)*$min_set_rx\s+
    (?:$hour_set_rx,)*$hour_set_rx\s+
    (?:$day_set_rx,)*$day_set_rx\s+
    (?:$month_set_rx,)*$month_set_rx\s+
    (?:$week_set_rx,)*$week_set_rx\s*
$/x;

my %non_production_crons = (moderation => {approve_everything => 1,}, automigrations => {apply_migrations => 1,},);

my $TEMPLATE = {
    all => {
        all => {
            fields => {
                attrs => {
                    fields => {
                        (map {$_ => {eq => TRUE, optional => TRUE, type => 'scalar',},} qw(deploy frontend lock)),
                        ttl => {optional => TRUE, regexp => qr/^\d+[hms]$/, type => 'scalar',},
                        (map {$_ => {optional => TRUE, type => 'int_un',},} qw(instances leasemem)),
                        stage => {
                            any_of => \@STAGES,
                            fields =>
                              {map {$_ => {eq => TRUE, optional => ($_ ne 'production'), type => 'scalar',},} @STAGES},
                            optional => TRUE,
                            type     => 'hash',
                        },
                        tag => {
                            all => {
                                len_min => 1,
                                type    => 'scalar',
                            },
                            optional => TRUE,
                            type     => 'array',
                        },
                        user => {
                            optional => TRUE,
                            regexp   => qr/^[a-z_][a-z0-9_-]{1,31}$/,
                            type     => 'scalar',
                        },
                        juggler_tag => {
                            optional => TRUE,
                            regexp   => qr/^[a-z0-9_-]{1,}$/,
                            type     => 'scalar',
                        },
                        frequency_limit => {
                            optional => TRUE,
                            regexp   => qr/^(?:1w|[1-9]\d*[dhm])$/,
                            type     => 'scalar',
                        },
                        deploy => {
                            optional => TRUE,
                            type     => 'boolean',
                        },
                    },
                    type => 'hash',
                },
                package => {
                    regexp => qr/^Cron::Methods::/,
                    type   => 'scalar',
                },
                sub  => {skip => TRUE,},
                time => {
                    regexp => $cron_rx,
                    type   => 'scalar',
                },
            },
            type => 'hash',
        },
        extra => TRUE,
        type  => 'hash',
    },
    extra => TRUE,
    type  => 'hash',
};

run_tests(
    sub {
        my ($app) = @_;

        my $methods = $app->get_cron_methods();

        for my $path (keys %$methods) {
            for my $method (keys %{$methods->{$path}}) {
                if ($non_production_crons{$path}{$method}) {
                    ok(!exists($methods->{$path}{$method}{attrs}{stage}{production}),
                        "$path $method has no production stage");
                    $methods->{$path}{$method}{attrs}{stage}{production} = 1;
                }
            }
        }

        my $qv = QBit::Validator->new(
            data     => $methods,
            template => $TEMPLATE,
        );
        ok(!$qv->has_errors(), 'Methods match the schema')
          or diag explain $qv->get_fields_with_error();
    },
    dont_create_database => 1,
    do_not_die_on_fail   => 1,
    application_package  => 'Cron',
);
