#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::More;
use Test::Partner::Utils;

use qbit;

my $PATH_NAME   = 'form';
my $METHOD_NAME = 'client_id';

plan tests => 7;

run_tests(
    sub {
        my ($app) = @_;

        subtest 'validation' => sub {
            plan tests => 2;

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => 'user_id=3.1415926',
                mode         => 'only_stdout',
            );

            is(
                $header,
"Status: 400 Bad request\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            is_deeply(
                from_json($out),
                {
                    'error_type' => 'Exception::Validation::BadArguments',
                    'message'    => 'user_id is not numeric: 3.1415926',
                    'result'     => 'error',
                },
                'exception'
            );
        };

        subtest 'user is in database' => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';
            local *{'Application::Model::Users::get_client_id'} = sub {fail "invalid call"};

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => 'user_id=1009',
                mode         => 'only_stdout',
            );

            is(
                $header,
                "Status: 200 OK\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            is_deeply(from_json($out)->{data}, {client_id => 11009}, 'got expected data',);
        };

        subtest 'user is not in database user is linked with client' => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';

            local *{'QBit::Application::Model::API::Yandex::Balance::find_client'} = sub {
                return [
                    {
                        agency_id      => 0,
                        city           => '',
                        client_id      => 1304746,
                        client_type_id => 0,
                    },
                ];
            };

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => 'user_id=35309619',
                mode         => 'only_stdout',
            );

            is(
                $header,
                "Status: 200 OK\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            is_deeply(from_json($out)->{data}, {client_id => 1304746}, 'got expected data',);
        };

        subtest 'db error' => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';

            local *{'Application::Model::Users::get'} = sub {return +{}};

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => 'user_id=35309619',
                mode         => 'only_stdout',
            );

            is(
                $header,
"Status: 500 Internal Error\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            is_deeply(
                from_json($out),
                {
                    'error_type' => 'Internal',
                    'message'    => 'Internal Error',
                    'result'     => 'error',
                },
                'exception'
            );
        };

        subtest 'balance error' => sub {
            plan tests => 2;

            no strict 'refs';
            no warnings 'redefine';

            local *{'Application::Model::Users::get'}                                       = sub {return undef};
            local *{'QBit::Application::Model::API::Yandex::Balance::get_client_id_by_uid'} = sub {return undef};
            local *{'QBit::Application::Model::API::Yandex::Balance::create_client'}        = sub {return 1304746};
            local *{'QBit::Application::Model::API::Yandex::Balance::call'}                 = sub {
                return [[-1, 'Some known error',],];
            };

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => 'user_id=35309619',
                mode         => 'only_stdout',
            );

            is(
                $header,
                "Status: 200 OK\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            is_deeply(
                from_json($out),
                {
                    'data' => {
                        'error' => {
                            'id'  => -1,
                            'txt' => 'Some known error',
                        },
                    },
                    'result' => 'ok',
                },
                'exception'
            );
        };

    },
    application_package => 'IntAPI',
    init                => [qw(api_balance intapi_acl)],
);
