#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::More tests => 4;
use Test::Differences;
use Test::Partner::Utils;
use Test::Partner2::Mock qw(mock_subs);

use qbit;

my $PATH_NAME   = 'form';
my $METHOD_NAME = 'create_or_update_partner';
my $FILE_NAME   = 'data.json';

run_tests(
    sub {
        my ($app) = @_;

        my $data = get_test_data_and_update_if_needed($FILE_NAME);     # Just read
        my $users = {map {$_->{id} => $_} @{clone $data->{logins}}};

        mock_subs(
            {
                'Application::Model::API::Yandex::AdFox::get_adfox_user_info' => sub {
                    return {'login' => 'mocked_login',};
                },

                'Application::Model::API::Yandex::AdFoxGraphQL::bind_user' => sub {
                    return {
                        id    => 123456,
                        login => '',
                    };
                },

                'Application::Model::API::Yandex::AdFoxGraphQL::create_user' => sub {
                    return {
                        id    => 123456,
                        login => 'mocked_adfox_login',
                    };
                },

                'QBit::Application::Model::API::Yandex::Balance::create_person' => sub {
                    return 1110;
                },

                'QBit::Application::Model::API::Yandex::Balance::create_offer' => sub {
                    # new contract ID + EXTERNAL_ID
                    return {'EXTERNAL_ID' => '111000/19', 'ID' => '1110222'};
                },

                'QBit::Application::Model::API::Yandex::Balance::get_client_persons' => sub {
                    # persons for the given client
                    return [
                        {id => 111},
                        {
                            id => 111666,
                            f0 => 123,
                            f1 => '123213w12',
                        }
                    ];
                },

                'Application::Model::Documents::get_contracts' => sub {
                    # active contract + person for the given user
                    return [
                        {
                            Collaterals => [],
                            Contract    => {
                                contract_type => 6,
                                dt            => '2000-01-01',
                            },
                            Person => {
                                client_id => 123,
                                id        => 111666,
                            },
                        }
                    ];
                },

                'Application::Model::Users::get_client_id' => sub {
                    return 123;
                },

                'QBit::Application::Model::API::Yandex::BlackBox::get_user_info' => sub {
                    my ($self, $user_id) = @_;

                    return $users->{$user_id};
                },

            }
        );

        subtest 'users state before registration' => sub {
            plan tests => scalar keys %$users;

            foreach my $user_id (keys %$users) {
                subtest sprintf('"%s" / %d', $users->{$user_id}{display_login}, $user_id) => sub {
                    eq_or_diff(get_user_with_all_rights($app, $user_id, fields => [qw(id login)]),
                        undef, 'No data in table users');

                    eq_or_diff(
                        $app->partner_db->user_role->get_all(
                            filter => {user_id => $user_id},
                            fields => [qw(user_id role_id)]
                        ),
                        [],
                        'No data in table user_role'
                    );
                  }
            }
        };

        subtest 'test registrations' => sub {
            plan tests => scalar @{$data->{tests}};

            foreach my $test (@{$data->{tests}}) {
                my %result;
                ($result{header}, $result{response}) = request_to_intapi(
                    app            => $app,
                    path           => $PATH_NAME,
                    method         => $METHOD_NAME,
                    request_method => 'POST',
                    content_type   => 'application/json',
                    query_string   => '',
                    mode           => 'split_json_answer',
                    body           => to_json($test->{body}),
                );

                #                ldump \%result;       # TODO remove
                if (ref $result{response} eq 'HASH' && $result{response}{result} && $result{response}{result} eq 'ok') {
                    my $user_id = $test->{body}{user_id};

                    $result{user} = get_user_with_all_rights(
                        $app, $user_id,
                        fields => [
                            qw(
                              client_id
                              country_id
                              has_common_offer
                              has_mobile_mediation
                              has_rsya
                              uid
                              is_assessor
                              is_efir_blogger
                              is_games
                              is_mobile_mediation
                              is_video_blogger
                              is_dm_lite
                              lastname
                              login
                              name
                              roles
                              inn
                              cooperation_form
                              self_employed
                              current_currency
                              features
                              )
                        ]
                    );
                }

                eq_or_diff(from_json(to_json(\%result)), $test->{result}, $test->{name});

                $test->{result} = \%result;    # for autoupdate
            }
        };

        get_test_data_and_update_if_needed($FILE_NAME, $data);    # Just save if needed
    },
    application_package => 'IntAPI',
    do_not_die_on_fail  => 1,
    init                => [qw(api_adfox api_adfox_graphql api_blackbox intapi_acl users rbac documents api_balance)],
);
