#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Differences;
use Test::More tests => 2 + 4;

use qbit;
use Test::Partner::Utils;
use Test::Partner2::Mock qw(mock_curdate mock_subs restore_subs);
use Test::Partner2::Simple;

my $PATH_NAME    = 'form';
my $METHOD_NAME  = 'form_version';
my $DATE         = '2021-11-11 20:21:11';
my $USER_ID      = 607369863;
my $KV_STORE_KEY = 'light_form_percent';

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate($DATE);

        # Получаем ошибку валидации
        # Ни в какую БД и кв-стор не ходим
        subtest 'validation' => sub {
            plan tests => 3;

            mock_subs(
                {
                    'QBit::Application::Model::DB::Table::get' => sub {
                        if (caller eq 'IntAPI::Method::Form') {
                            fail 'table get';
                        } else {
                            return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::Table::get'}
                              ->(@_);
                        }
                    },
                    'QBit::Application::Model::KvStore::get' => sub {
                        if (caller eq 'IntAPI::Method::Form') {
                            fail 'kv_store get';
                        } else {
                            return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::KvStore::get'}
                              ->(@_);
                        }
                    },
                    'QBit::Application::Model::DB::mysql::Table::add' => sub {
                        if (caller eq 'IntAPI::Method::Form') {
                            fail 'table add';
                        } else {
                            return $Test::Partner2::Mock::original_subs->{
                                'QBit::Application::Model::DB::mysql::Table::add'}->(@_);
                        }
                    },
                }
            );

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => 'user_id=3.1415926',
                mode         => 'only_stdout',
            );

            is(
                $header,
"Status: 400 Bad request\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            eq_or_diff(
                from_json($out),
                {
                    error_type => 'Exception::Validation::BadArguments',
                    message    => 'user_id is not numeric: 3.1415926',
                    result     => 'error',
                },
                'exception'
            );
            eq_or_diff($app->partner_db->form_version->get_all(), [], 'Empty DB data');

            restore_subs(
                [
                    qw(
                      QBit::Application::Model::DB::Table::get
                      QBit::Application::Model::KvStore::get
                      QBit::Application::Model::DB::mysql::Table::add
                      )
                ]
            );
        };

        # Валидация успешна
        # Идём в БД за кешем, кеша нет
        # Идём за вероятностью в кв-стор
        # Записываем кеш
        subtest 'version with 0% probability' => sub {
            plan tests => 7;

            mock_subs(
                {
                    'QBit::Application::Model::DB::Table::get' => sub {
                        my ($self, $pk, %opts) = @_;

                        if (caller eq 'IntAPI::Method::Form') {
                            is($pk, $USER_ID, 'table get pk');
                            eq_or_diff \%opts, {fields => ['version']}, 'table get opts';
                        }
                        return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::Table::get'}->(@_);
                    },
                    'QBit::Application::Model::KvStore::get' => sub {
                        my ($self, $key) = @_;

                        if (caller eq 'IntAPI::Method::Form') {
                            is $key, $KV_STORE_KEY, 'kv_store get';
                            return undef;
                        } else {
                            return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::KvStore::get'}
                              ->(@_);
                        }
                    },
                    'QBit::Application::Model::DB::mysql::Table::add' => sub {
                        my ($self, $data) = @_;

                        if (caller eq 'IntAPI::Method::Form') {
                            eq_or_diff $data,
                              {
                                dt      => $DATE,
                                user_id => $USER_ID,
                                version => 1,
                              },
                              'table add';
                        }
                        return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::mysql::Table::add'}
                          ->(@_);
                    },
                }
            );

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => sprintf('user_id=%d', $USER_ID),
                mode         => 'only_stdout',
            );

            is(
                $header,
                "Status: 200 OK\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            eq_or_diff(from_json($out)->{data}, {version => 1}, 'got expected data',);
            eq_or_diff(
                $app->partner_db->form_version->get_all(),
                [
                    {
                        dt      => $DATE,
                        user_id => $USER_ID,
                        version => 1,
                    }
                ],
                'Correct DB data'
            );

            restore_subs(
                [
                    qw(
                      QBit::Application::Model::DB::Table::get
                      QBit::Application::Model::KvStore::get
                      QBit::Application::Model::DB::mysql::Table::add
                      )
                ]
            );
        };

        # Валидация успешна
        # Идём в БД за кешем, кеша есть (от прошлого теста остался)
        # В кв-стор не ходим и кеш не перезаписываем
        subtest 'version with 0% probability, cached' => sub {
            plan tests => 4;

            mock_subs(
                {
                    'QBit::Application::Model::DB::Table::get' => sub {
                        my ($self, $pk, %opts) = @_;

                        if (caller eq 'IntAPI::Method::Form') {
                            is($pk, $USER_ID, 'table get pk');
                            eq_or_diff \%opts, {fields => ['version']}, 'table get opts';
                        }
                        return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::Table::get'}->(@_);
                    },
                    'QBit::Application::Model::KvStore::get' => sub {
                        if (caller eq 'IntAPI::Method::Form') {
                            fail 'kv_store get';
                        } else {
                            return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::KvStore::get'}
                              ->(@_);
                        }
                    },
                    'QBit::Application::Model::DB::mysql::Table::add' => sub {
                        if (caller eq 'IntAPI::Method::Form') {
                            fail 'table add';
                        } else {
                            return $Test::Partner2::Mock::original_subs->{
                                'QBit::Application::Model::DB::mysql::Table::add'}->(@_);
                        }
                    },
                }
            );

            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => sprintf('user_id=%d', $USER_ID),
                mode         => 'only_stdout',
            );

            is(
                $header,
                "Status: 200 OK\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            eq_or_diff(from_json($out)->{data}, {version => 1}, 'got expected data',);

            restore_subs(
                [
                    qw(
                      QBit::Application::Model::DB::Table::get
                      QBit::Application::Model::KvStore::get
                      QBit::Application::Model::DB::mysql::Table::add
                      )
                ]
            );

            # Чтобы в следующем тесте пересчитать без кеша
            $app->partner_db->form_version->delete($app->partner_db->filter(\1));
        };

        # Валидация успешна
        # Идём в БД за кешем, кеша нет
        # Идём за вероятностью в кв-стор
        # Записываем кеш
        subtest 'version with 100% probability' => sub {
            plan tests => 7;

            mock_subs(
                {
                    'QBit::Application::Model::DB::Table::get' => sub {
                        my ($self, $pk, %opts) = @_;

                        if (caller eq 'IntAPI::Method::Form') {
                            is($pk, $USER_ID, 'table get pk');
                            eq_or_diff \%opts, {fields => ['version']}, 'table get opts';
                        }
                        return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::Table::get'}->(@_);
                    },
                    'QBit::Application::Model::KvStore::get' => sub {
                        my ($self, $key) = @_;

                        if (caller eq 'IntAPI::Method::Form') {
                            is $key, $KV_STORE_KEY, 'kv_store get';
                            return 100;
                        } else {
                            return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::KvStore::get'}
                              ->(@_);
                        }
                    },
                    'QBit::Application::Model::DB::mysql::Table::add' => sub {
                        my ($self, $data) = @_;

                        if (caller eq 'IntAPI::Method::Form') {
                            eq_or_diff $data,
                              {
                                dt      => $DATE,
                                user_id => $USER_ID,
                                version => 2,
                              },
                              'table add';
                        }
                        return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::mysql::Table::add'}
                          ->(@_);
                    },
                }
            );
            my ($header, $out) = request_to_intapi(
                app          => $app,
                path         => $PATH_NAME,
                method       => $METHOD_NAME,
                query_string => sprintf('user_id=%d', $USER_ID),
                mode         => 'only_stdout',
            );

            is(
                $header,
                "Status: 200 OK\nX-Content-Type-Options: nosniff\nContent-Type: application/json; charset=UTF-8",
                'got expected header'
              );
            eq_or_diff(from_json($out)->{data}, {version => 2}, 'got expected data',);
            eq_or_diff(
                $app->partner_db->form_version->get_all(),
                [
                    {
                        dt      => $DATE,
                        user_id => $USER_ID,
                        version => 2,
                    }
                ],
                'Correct DB data'
            );

            restore_subs(
                [
                    qw(
                      QBit::Application::Model::DB::Table::get
                      QBit::Application::Model::KvStore::get
                      QBit::Application::Model::DB::mysql::Table::add
                      )
                ]
            );
        };

    },
    application_package => 'IntAPI',
    init                => [qw(kv_store partner_db)],
);
