#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use PiConstants qw($IDM_SLUG $TVM_HTTP_HEADER_NAME);
use Test::Partner2::Simple;
use Test::Partner::Utils;
use Test::Differences;
use Test::More;

use Test::Partner2::Mock qw(mock_subs);

use qbit;

my @result;

run_tests(
    sub {
        my ($app) = @_;
        mock_curdate('2019-09-03 15:00:00');

        my $tvm_idm          = $app->get_option('tvm_idm');
        my $get_uid_by_login = sub {
            my ($self, $login) = @_;
            throw "There is no info about login '$login' in blackbox";
        };
        mock_subs(
            {
                'IntAPI::Method::get_tvm_source' => sub {
                    return $tvm_idm;
                },
                'QBit::Application::Model::API::Yandex::BlackBox::get_uid_by_login' => sub {
                    $get_uid_by_login->(@_);
                },
                'Application::Model::API::Yandex::Staff::persons' => sub {
                    return {
                        "name" => {
                            "first" => {"en" => "mocked"},
                            "last"  => {"en" => "user"},
                        },
                        "work_email" => 'mocked-user@yandex-team.ru'
                    };
                },
                'Application::Model::Users::get_client_id' => sub {
                    return 1000_000_000;
                },
                'Utils::Logger::_log' => sub {
                    my (@p) = @_;
                    push @{$app->{_TO_LOG_}}, {log => \@p};
                },
                'Utils::MonitoringUtils::send_to_graphite' => sub {
                    my (%p) = @_;
                    push @{$app->{_TO_LOG_}}, {graphite => \%p};
                },
                'Application::Model::MailNotification::add' => sub {
                    my ($self, %p) = @_;
                    push @{$app->{_TO_LOG_}}, {mail => \%p};
                },
            }
        );

        push @result,
          {
            '00_test_name' => 'roles before',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => 1014})
          };

        make_request(
            $app, 'no passport login',
            'mocked-yan-manager', {$IDM_SLUG => "YAN_VIEWER"},
            "/$IDM_SLUG/YAN_VIEWER", {}
        );
        make_request(
            $app, 'unexists user',
            'unexists-user',         {$IDM_SLUG        => "YAN_VIEWER"},
            "/$IDM_SLUG/YAN_VIEWER", {'passport-login' => 'unexists-user'}
        );
        make_request($app, 'bad slug', 'mocked-dsp-manager', {$IDM_SLUG => "YAN_VIEWER"},
            "/{$IDM_SLUG}_bad/YAN_VIEWER", {'passport-login' => 'mocked-dsp-manager'});
        make_request($app, 'diff role', 'mocked-dsp-manager', {$IDM_SLUG => "YAN_VIEWER"},
            "/$IDM_SLUG/YAN_MANAGER", {'passport-login' => 'mocked-dsp-manager'});
        make_request(
            $app, 'incompartible roles',
            'mocked-yan-manager',    {$IDM_SLUG        => "YAN_VIEWER"},
            "/$IDM_SLUG/YAN_VIEWER", {'passport-login' => 'mocked-yan-manager'}
        );
        make_request($app, 'robot role', 'mocked-yan-manager', {$IDM_SLUG => "ROBOT_ASSISTANT"},
            "/$IDM_SLUG/ROBOT_ASSISTANT", {'passport-login' => 'mocked-yan-manager'});
        make_request(
            $app, 'internal to partner',
            'mocked-yan-partner',    {$IDM_SLUG        => "YAN_VIEWER"},
            "/$IDM_SLUG/YAN_VIEWER", {'passport-login' => 'mocked-yan-partner'}
        );
        make_request(
            $app, 'already has role',
            'mocked-dsp-manager',     {$IDM_SLUG        => "DSP_MANAGER"},
            "/$IDM_SLUG/DSP_MANAGER", {'passport-login' => 'mocked-dsp-manager'}
        );
        make_request($app, 'success', 'mocked-dsp-manager', {$IDM_SLUG => "YAN_VIEWER"},
            "/$IDM_SLUG/YAN_VIEWER", {'passport-login' => 'mocked-dsp-manager'});

        push @result,
          {
            '00_test_name' => 'roles after',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => 1014})
          };

        $get_uid_by_login = sub {
            return 1000_000_000;
        };

        make_request(
            $app, 'create mocked-user',
            'mocked-user',            {$IDM_SLUG        => "YAN_MANAGER"},
            "/$IDM_SLUG/YAN_MANAGER", {'passport-login' => 'mocked-user'}
        );

        my $get_all_result = $app->partner_db->users->get_all(filter => {uid => 1000000000});
        push @result,
          {
            '00_test_name' => 'roles mocked-user',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => $get_all_result->[0]{id}})
          };
        push @result,
          {
            '00_test_name' => 'data mocked-user',
            '01_data'      => $get_all_result,
          };

        push @result,
          {
            '00_test_name' => 'pages mocked-dsp-manager before',
            '01_data'      => $app->partner_db->managers->get_all(filter => {manager_id => 1014})
          };

        make_request(
            $app, 'without page_id',
            'mocked-dsp-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-dsp-manager'}
        );

        make_request(
            $app, 'with bad page_id',
            'mocked-dsp-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-dsp-manager', 'page_id' => "PAGE"}
        );

        make_request(
            $app, 'with external page_id',
            'mocked-dsp-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-dsp-manager', 'page_id' => 41443}
        );

        make_request(
            $app, 'with page_id', 'mocked-dsp-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-dsp-manager', 'page_id' => 2}
        );

        make_request(
            $app, 'with dublicate page_id',
            'mocked-dsp-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-dsp-manager', 'page_id' => 2}
        );

        push @result,
          {
            '00_test_name' => 'pages mocked-dsp-manager after',
            '01_data'      => $app->partner_db->managers->get_all(filter => {manager_id => 1014})
          };

        push @result,
          {
            '00_test_name' => 'roles after new',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => 1014})
          };

        my $has = get_test_data_and_update_if_needed('data.json', \@result);
        eq_or_diff(\@result, $has, 'body');

    },
    application_package => 'IntAPI',
    do_not_die_on_fail  => TRUE,
    init                => ['api_blackbox', 'api_staff', 'users', 'mail_notification'],
    mocks => {mock_send_to_graphite => FALSE},
);

sub make_request {
    my ($app, $test_name, $login, $role, $path, $fields) = @_;

    $app->{_TO_LOG_} = [];

    my ($header, $body) = request_to_intapi(
        app          => $app,
        path         => 'idm',
        method       => 'add_role',
        mode         => 'split_json_answer',
        query_string => "login=$login" . "&role=" . to_json($role) . "&fields=" . to_json($fields) . "&path=$path",
        body         => '',
        http_headers => {$TVM_HTTP_HEADER_NAME => 'valid_ticket'},
    );

    push @result,
      {
        '00_test_name' => $test_name,
        '01_input'     => {login => $login, role => $role, path => $path, fields => $fields},
        '02_headers' => [split /\n/, $header],
        '03_body '   => $body,
        '04_log'     => $app->{_TO_LOG_},
      };
}
