#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use PiConstants qw($IDM_SLUG $TVM_HTTP_HEADER_NAME);
use Test::Partner2::Simple;
use Test::Partner::Utils;
use Test::Differences;
use Test::More;

use Test::Partner2::Mock qw(mock_subs);

use qbit;

my @result;

run_tests(
    sub {
        my ($app) = @_;

        my $tvm_idm = $app->get_option('tvm_idm');
        mock_subs(
            {
                'IntAPI::Method::get_tvm_source' => sub {
                    return $tvm_idm;
                },
                'Utils::Logger::_log' => sub {
                    my (@p) = @_;
                    push @{$app->{_TO_LOG_}}, {log => \@p,};
                },
                'Utils::MonitoringUtils::send_to_graphite' => sub {
                    my (%p) = @_;
                    push @{$app->{_TO_LOG_}}, {graphite => \%p,};
                },
                'Application::Model::MailNotification::add' => sub {
                    my ($self, %p) = @_;
                    push @{$app->{_TO_LOG_}}, {mail => \%p};
                  }
            }
        );

        push @result,
          {
            '00_test_name' => 'roles before',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => 668991881})
          };

        make_request(
            $app, 'no passport login',
            'mocked-yan-manager', {$IDM_SLUG => "YAN_VIEWER"},
            "/$IDM_SLUG/YAN_VIEWER", {}
        );
        make_request(
            $app, 'unexists user',
            'unexists-user',          {$IDM_SLUG        => "YAN_MANAGER"},
            "/$IDM_SLUG/YAN_MANAGER", {'passport-login' => 'unexists-user'}
        );
        make_request($app, 'bad slug', 'mocked-yndx-adfox', {$IDM_SLUG => "YAN_MANAGER"},
            "/bad$IDM_SLUG/YAN_MANAGER", {'passport-login' => 'mocked-yndx-adfox'});
        make_request($app, 'diff roles', 'mocked-yndx-adfox', {$IDM_SLUG => "YAN_MANAGER"},
            "/$IDM_SLUG/YAN_MANAGER2", {'passport-login' => 'mocked-yndx-adfox'});
        make_request(
            $app, 'unexists roles',
            'mocked-yndx-adfox',      {$IDM_SLUG        => "DSP_MANAGER"},
            "/$IDM_SLUG/DSP_MANAGER", {'passport-login' => 'mocked-yndx-adfox'}
        );
        make_request($app, 'success', 'mocked-yndx-adfox', {$IDM_SLUG => "YAN_MANAGER"},
            "/$IDM_SLUG/YAN_MANAGER", {'passport-login' => 'mocked-yndx-adfox'});

        push @result,
          {
            '00_test_name' => 'roles after',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => 668991881})
          };

        push @result,
          {
            '00_test_name' => 'roles mocked-internal-manager before',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => 1023})
          };

        make_request(
            $app, 'without page_id',
            'mocked-internal-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-internal-manager'}
        );

        make_request(
            $app, 'invalid page_id',
            'mocked-internal-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-internal-manager', 'page_id' => "PAGE"}
        );

        make_request(
            $app, 'wrong page_id',
            'mocked-internal-manager',
            {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
            "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
            {'passport-login' => 'mocked-internal-manager', 'page_id' => 1}
        );

        push @result,
          {
            '00_test_name' => 'pages mocked-internal-manager before',
            '01_data'      => $app->partner_db->managers->get_all(filter => {manager_id => 1023})
          };

        for my $page_id (qw(2 88848 88849 118826 128843 132439)) {
            make_request(
                $app, 'page_id ' . $page_id,
                'mocked-internal-manager',
                {$IDM_SLUG => "INTERNAL_YAN_MANAGER"},
                "/$IDM_SLUG/INTERNAL_YAN_MANAGER",
                {'passport-login' => 'mocked-internal-manager', 'page_id' => $page_id}
            );
        }

        push @result,
          {
            '00_test_name' => 'pages mocked-internal-manager after',
            '01_data'      => $app->partner_db->managers->get_all(filter => {manager_id => 1023})
          };

        push @result,
          {
            '00_test_name' => 'roles mocked-internal-manager after',
            '01_data'      => $app->partner_db->user_role->get_all(filter => {user_id => 1023})
          };

        my $has = get_test_data_and_update_if_needed('data.json', \@result);
        eq_or_diff(\@result, $has, 'body');

    },
    application_package => 'IntAPI',
    do_not_die_on_fail  => TRUE,
    init                => ['mail_notification'],
    mocks               => {mock_send_to_graphite => FALSE},
);

sub make_request {
    my ($app, $test_name, $login, $role, $path, $fields, $fired) = @_;
    $fired //= '';

    $app->{_TO_LOG_} = [];

    my ($header, $body) = request_to_intapi(
        app          => $app,
        path         => 'idm',
        method       => 'remove_role',
        mode         => 'split_json_answer',
        query_string => "login=$login"
          . "&fields="
          . to_json($fields)
          . "&role="
          . to_json($role)
          . "&path=$path"
          . "&fired=$fired",
        body         => '',
        http_headers => {$TVM_HTTP_HEADER_NAME => 'valid_ticket'},
    );

    push @result,
      {
        '00_test_name' => $test_name,
        '01_input'     => {login => $login, role => $role, path => $path, fired => $fired, fields => $fields},
        '02_headers' => [split /\n/, $header],
        '03_body '   => $body,
        '04_log'     => $app->{_TO_LOG_},
      };
}
