#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use File::Path qw(make_path);
use File::Slurp qw(read_file  write_file);
use File::Temp qw(tempfile);
use Test::Differences qw(eq_or_diff);

use Test::More;
use Test::Partner2::Simple;
use Test::Partner2::Mock qw( mock_time );
require QBit::Application::Model::API::HTTP;

use qbit;

my $DEFAULT_CONTENT = "some\ncontent";
my $CACHED_CONTENT  = "some\ncached\ncontent";

my $CACHED_PATH;
run_tests(
    sub {
        my ($app) = @_;

        my $tmp_filepath = '';
        {
            File::Path::make_path('/tmp/call_check');
            (undef, $tmp_filepath) = File::Temp::tempfile(
                'check_data_XXXXX',
                UNLINK => 1,
                OPEN   => 0,
                SUFFIX => '.dat',
                TMPDIR => 1
            );
        }

        # Setup
        $CACHED_PATH = $app->get_option('chi_file_cache_dir');
        unlink glob "$CACHED_PATH/*_${$}_*";
        unlink $tmp_filepath;

        mock_time('2018-04-10 00:00:00', '%Y-%m-%d %H:%M:%S');

        my $fixture = {
            'GET - simple' => {
                options => {url => 'http://foo?bar=1'},
                params  => {},
                expect  => $DEFAULT_CONTENT
            },
            'GET - :return_ref' => {
                options => {url           => 'http://foo?bar=1'},
                params  => {':return_ref' => 1},
                expect  => \$DEFAULT_CONTENT
            },
            'GET - :return_fh got error without :cached' => {
                options => {url          => 'http://foo?bar=1'},
                params  => {':return_fh' => 1},
                error   => 'Param ":return_fh" cannot be used without ":cached"'
            },

            'GET - :content_file' => {
                options             => {url             => 'http://foo?bar=1'},
                method              => '',
                params              => {':content_file' => $tmp_filepath},
                expect              => '',
                expect_content_file => $DEFAULT_CONTENT
            },

            'GET - error no method with :cached' => {
                options => {url       => 'http://foo?bar=1'},
                params  => {':cached' => 1,},
                error   => 'Param "method" cannot be empty with ":cached"',
            },

            'GET - :cached no cache' => {
                options => {url       => 'http://foo'},
                method  => 'add',
                params  => {':cached' => 1,},
                expect  => $DEFAULT_CONTENT
            },
            'GET - :cached no cache :return_ref' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'     => 1,
                    ':return_ref' => 1
                },
                expect => \$DEFAULT_CONTENT
            },
            'GET - :cached no cache :return_fh' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'    => 1,
                    ':return_fh' => 1,
                },
                expect => $DEFAULT_CONTENT
            },

            'GET - :cached no cache :content_file' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'       => 1,
                    ':content_file' => $tmp_filepath
                },
                expect              => '',
                expect_content_file => $DEFAULT_CONTENT
            },
            'GET - :cached no cache :content_file :return_ref' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'       => 1,
                    ':content_file' => $tmp_filepath,
                    ':return_ref'   => 1
                },
                error => 'Param ":content_file" cannot be used with ":return_fh" and ":return_ref"',
            },
            'GET - :cached no cache :content_file :return_fh' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'       => 1,
                    ':content_file' => $tmp_filepath,
                    ':return_fh'    => 1,
                },
                error => 'Param ":content_file" cannot be used with ":return_fh" and ":return_ref"',
            },

            'GET - :cached from cache' => {
                options => {url => 'http://foo'},
                method  => 'met/h.od',
                params  => {
                    ':cached' => 1,
                    foo       => 1,
                    bar       => 1,
                },
                add_files => [
                    {
                        path    => _get_cache_file_path('met_h_od', 'bar_1_foo_1'),
                        content => $CACHED_CONTENT
                    }
                ],
                expect => $CACHED_CONTENT
            },
            'GET - :cached  from cache :return_ref' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'     => 1,
                    ':return_ref' => 1,
                },
                add_files => [
                    {
                        path    => _get_cache_file_path('add'),
                        content => $CACHED_CONTENT
                    }
                ],
                expect => \$CACHED_CONTENT
            },
            'GET - :cached  from cache :return_fh' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'    => 1,
                    ':return_fh' => 1,
                },
                add_files => [
                    {
                        path    => _get_cache_file_path('add'),
                        content => $CACHED_CONTENT
                    }
                ],
                expect => $CACHED_CONTENT
            },

            'GET - :cached from cache :content_file' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'       => 1,
                    ':content_file' => $tmp_filepath
                },
                add_files => [
                    {
                        path    => $tmp_filepath,
                        content => $CACHED_CONTENT
                    },
                ],
                expect              => '',
                expect_content_file => $CACHED_CONTENT
            },
            'GET - :cached  from cache :content_file :return_ref' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'       => 1,
                    ':return_ref'   => 1,
                    ':content_file' => $tmp_filepath
                },
                add_files => [
                    {
                        path    => $tmp_filepath,
                        content => $CACHED_CONTENT
                    }
                ],
                error => 'Param ":content_file" cannot be used with ":return_fh" and ":return_ref"',
            },
            'GET - :cached  from cache :content_file :return_fh' => {
                options => {url => 'http://foo'},
                method  => 'add',
                params  => {
                    ':cached'       => 1,
                    ':return_fh'    => 1,
                    ':content_file' => $tmp_filepath
                },
                add_files => [
                    {
                        path    => $tmp_filepath,
                        content => $CACHED_CONTENT
                    }
                ],
                error => 'Param ":content_file" cannot be used with ":return_fh" and ":return_ref"',
            },
            'GET - :memcached got error with :cached' => {
                options => {url          => 'http://foo?bar=1'},
                params  => {':memcached' => 1, ':cached' => 1},
                error => 'Param ":memcached" cannot be used with ":cached", ":content_file" and ":return_fh"'
            },
            'GET - :memcached got error with :content_file' => {
                options => {url          => 'http://foo?bar=1'},
                params  => {':memcached' => 1, ':content_file' => 1},
                error => 'Param ":memcached" cannot be used with ":cached", ":content_file" and ":return_fh"'
            },
            'GET - :memcached got error with :return_fh' => {
                options => {url          => 'http://foo?bar=1'},
                params  => {':memcached' => 1, ':return_fh' => 1},
                error => 'Param ":memcached" cannot be used with ":cached", ":content_file" and ":return_fh"'
            },
            'GET - :memcached no cache' => {
                options   => {url          => 'http://foo?bar=1'},
                params    => {':memcached' => 1},
                expect    => $DEFAULT_CONTENT,
                memcached => {
                    key       => ['api_adfox', 'http://foo?bar=1'],
                    get_value => undef,
                    set_value => to_json({data => $DEFAULT_CONTENT}),
                    to        => 1,
                },
            },
            'GET - :memcached has cache' => {
                options   => {url          => 'http://foo?bar=1'},
                params    => {':memcached' => 1},
                expect    => $CACHED_CONTENT,
                memcached => {
                    key => ['api_adfox', 'http://foo?bar=1'],
                    get_value => to_json({data => $CACHED_CONTENT}),
                    set_value => to_json({data => $CACHED_CONTENT}),
                    to        => 1,
                },
            },
        };

        my $memcache_content;
        {
            no warnings 'redefine';
            no strict 'refs';

            *{'LWP::UserAgent::request'} = sub {
                my ($mock, $request, $content_file_path) = @_;

                my $r = HTTP::Response->new(200);
                $r->request($request);
                $r->content($DEFAULT_CONTENT);

                if ($content_file_path) {
                    $r->content('');
                    write_file($content_file_path, $DEFAULT_CONTENT);
                } else {
                    $r->content($DEFAULT_CONTENT);
                }

                return $r;
            };
            *{'QBit::Application::Model::Memcached::get'} = sub {
                my ($self, $prefix, $key) = @_;
                eq_or_diff([$prefix, $key], $memcache_content->{key}, "valid key for get");
                return $memcache_content->{get_value};
            };
            *{'QBit::Application::Model::Memcached::set'} = sub {
                my ($self, $prefix, $key, $value, $to) = @_;
                eq_or_diff([$prefix, $key], $memcache_content->{key}, "valid key for set");
                eq_or_diff($to,    $memcache_content->{to},        "valid to for set");
                eq_or_diff($value, $memcache_content->{set_value}, "valid value for set");
            };
        }

        my $self = $app->api_adfox;
        foreach my $testname (sort keys %$fixture) {
            my $test_data = $fixture->{$testname};

            $self->set_option($_, $test_data->{options}->{$_}) for keys %{$test_data->{options} // {}};

            if ($test_data->{add_files}) {
                foreach my $file_data (@{$test_data->{add_files}}) {
                    File::Slurp::write_file($file_data->{path}, $file_data->{content});
                }
            }

            if ($test_data->{memcached}) {
                $memcache_content = $test_data->{memcached};
            }

            my $got     = undef;
            my $err_msg = '';
            try {
                $got = QBit::Application::Model::API::HTTP::call($self, $test_data->{method}, %{$test_data->{params}});
            }
            catch {
                my ($exception) = @_;
                $err_msg = $exception->message // '';
                $err_msg =~ s/ at .*$//;
            };

            subtest $testname => sub {
                note("Start test $testname");

                my $expect_err = $test_data->{error};
                if (defined $expect_err) {
                    eq_or_diff($err_msg, $expect_err, "got error");
                } elsif ($err_msg && !$expect_err) {
                    eq_or_diff($err_msg, '', "got no error");
                }

                my $expect = $test_data->{expect};
                if (defined $expect) {
                    if ($test_data->{params}->{':return_fh'}) {
                        cmp_ok(ref($got), 'eq', 'GLOB', 'got a GLOB');
                        local $/ = undef;    # slurp mode
                        $got = <$got>;
                    }

                    eq_or_diff($got, $expect, "content");
                }

                if (exists $test_data->{expect_content_file}) {
                    my $got_content_file = undef;
                    my $file_path        = $test_data->{params}->{':content_file'};
                    if (-f $file_path) {
                        $got_content_file = File::Slurp::read_file($file_path);
                    }

                    eq_or_diff($got_content_file, $test_data->{expect_content_file}, "file content");
                }
            };

            # teardown
            $self->set_option($_, undef) for keys %{$test_data->{options} // {}};
            unlink glob "$CACHED_PATH/*_${$}_*";
            unlink $tmp_filepath;
            $memcache_content = undef;
        }
    },
    dont_create_database => 1,
    init                 => ['memcached'],
);

sub _get_cache_file_path {
    my ($method_escaped, $opts_string) = @_;

    my $proc_id = $$;
    my $proc_datetime = curdate(oformat => 'db_time');

    $proc_datetime =~ s/:/-/g;
    $proc_datetime =~ s/ /_/;

    my $cache_key = join '_', $proc_datetime, $proc_id, $method_escaped, $opts_string // '';
    my $cache_file_path = sprintf '%s/%s.dat', $CACHED_PATH, $cache_key;

    return $cache_file_path;
}
