#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::Differences qw(eq_or_diff);

use qbit;
use Test::Partner2::Simple;
use Test::Partner::Utils;

require QBit::Application::Model::API::HTTP;

my $fixture = {
    'GET - simple' => {
        options => {url => 'http://foo?bar=1'},
        method  => '',
        expect  => q[GET http://foo?bar=1

]
    },
    'GET - simple 2' => {
        options => {url   => 'http://foo'},
        method  => '',
        params  => {'bar' => 1,},
        expect  => q[GET http://foo?bar=1

]
    },
    'GET - simple 3' => {
        options => {url => 'http://foo'},
        method  => '',
        params  => {
            'bar'  => 1,
            ':get' => 1,
        },
        expect => q[GET http://foo?bar=1

]
    },
    'GET - simple 4' => {
        options => {url      => 'http://foo/'},
        method  => 'bar',
        params  => {'foobar' => 1,},
        expect  => q[GET http://foo/bar?foobar=1

]
    },
    'GET - with white list headers' => {
        options => {
            url     => 'http://foo',
            headers => {'Host' => 'foo.bar',}
        },
        params => {
            'bar'           => 1,
            ':content_type' => 'application/json',
        },
        expect => q[GET http://foo?bar=1
Host: foo.bar
Content-Type: application/json

]
    },
    'GET - with POSTDATA' => {
        options => {url        => 'http://foo/'},
        method  => '',
        params  => {':content' => "foo\nbar"},
        expect  => q[GET http://foo/
Content-Length: 7

foo
bar
]
    },

    'GET - error for unknown header' => {
        options => {
            url     => 'http://foo?bar=1',
            headers => {
                'Host'         => 'foo.bar',
                'Content-type' => 'text/plain'
            }
        },
        params => {'bar' => 1,},
        error  => 'Bad config for headers: Content-type'
    },

    'POST - simple' => {
        options => {url     => 'http://foo/'},
        method  => '',
        params  => {':post' => 1,},
        expect  => q[POST http://foo/
Content-Length: 0
Content-Type: application/x-www-form-urlencoded

]
    },
    'POST - simple 2' => {
        options => {url     => 'http://foo?bar=1'},
        method  => '',
        params  => {':post' => 1,},
        expect  => q[POST http://foo?bar=1
Content-Length: 0
Content-Type: application/x-www-form-urlencoded

]
    },
    'POST - with POSTDATA' => {
        options => {url => 'http://foo/'},
        method  => '',
        params  => {
            ':post'    => 1,
            ':content' => "foo\nbar"
        },
        expect => q[POST http://foo/
Content-Length: 7
Content-Type: application/x-www-form-urlencoded

foo
bar
]
    },
    'POST - with form-urlencoded params' => {
        options => {url => 'http://foo/'},
        method  => '',
        params  => {
            ':post' => 1,
            foo     => 'bar',
            bar     => 'baz'
        },
        expect => q[POST http://foo/
Content-Length: 15
Content-Type: application/x-www-form-urlencoded

bar=baz&foo=bar
]
    },
    'POST - with POSTDATA form-urlencoded' => {
        options => {url => 'http://foo/'},
        method  => '',
        params  => {
            ':post'    => 1,
            ':content' => {
                foo => 'bar',
                bar => 'baz'
            }
        },
        expect => q[POST http://foo/
Content-Length: 15
Content-Type: application/x-www-form-urlencoded

bar=baz&foo=bar
]
    },
    'POST - with white list headers' => {
        options => {
            url     => 'http://foo?bar=1',
            headers => {'Host' => 'foo.bar'}
        },
        params => {
            ':post'         => 1,
            ':content_type' => 'application/json',
            foo             => 'bar',
            bar             => 'baz'
        },
        expect => q[POST http://foo?bar=1
Host: foo.bar
Content-Length: 15
Content-Type: application/json

bar=baz&foo=bar
]
    },
    'POST - with unknown header' => {
        options => {
            url     => 'http://foo?bar=1',
            headers => {
                'Host'         => 'foo.bar',
                'Content-type' => 'text/plain'
            }
        },
        params => {':post' => 1,},
        error  => 'Bad config for headers: Content-type'
    },

};

run_tests(
    sub {
        my ($app) = @_;

        my $LWPRequest = undef;
        {
            no warnings 'redefine';
            no strict 'refs';

            *{'LWP::UserAgent::request'} = sub {
                my ($mock, $request) = @_;

                $LWPRequest = $request;

                my $r = HTTP::Response->new(200);
                $r->request($request);
                $r->content('foo-content');
                return $r;
            };

        }

        my $self = $app->api_adfox;
        foreach my $testname (sort keys %$fixture) {
            my $test_data = $fixture->{$testname};

            $self->set_option($_, $test_data->{options}->{$_}) for keys %{$test_data->{options} // {}};

            my $err_msg = '';

            try {
                QBit::Application::Model::API::HTTP::call($self, $test_data->{method}, %{$test_data->{params}});
            }
            catch {
                my ($exception) = @_;
                $err_msg = $exception->message // '';
                $err_msg =~ s/ at .*$//;
            };

            my $expect_err = $test_data->{error};
            if (defined $expect_err) {
                eq_or_diff($err_msg, $expect_err, "$testname");
            } elsif ($err_msg && !$expect_err) {
                eq_or_diff($err_msg, '', "$testname got no error");
            }

            my $got = $LWPRequest ? $LWPRequest->as_string : '';
            my $expect = $test_data->{expect};
            eq_or_diff_http_message($got, $expect, "$testname", {context => 10}) if defined $expect;

            # teardown
            $LWPRequest = undef;
            $self->set_option($_, undef) for keys %{$test_data->{options} // {}};
        }
    },
    dont_create_database => 1,
);
