#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Log::Log4perl qw(get_logger :levels);
use Test::Differences qw(eq_or_diff);

use qbit;
use Test::Partner2::Simple;
require QBit::Application::Model::API::HTTP;

my $fixture = {
    'GET - simple' => {
        uri    => 'http://foo?bar=1',
        params => {':get' => 1,},
        expect => q[GET http://foo?bar=1

]
    },
    'GET - with params' => {
        uri    => 'http://foo',
        params => {
            ':get' => 1,
            'foo'  => 'bar'
        },
        expect => q[GET http://foo?foo=bar

]
    },
    'GET - with white list headers' => {
        uri     => 'http://foo?bar=1',
        options => {headers => {'Host' => 'foo.bar',}},
        params  => {
            ':get'          => 1,
            ':content_type' => 'application/json',
        },
        expect => q[GET http://foo?bar=1
Host: foo.bar
Content-Type: application/json

]
    },
    'GET - with unknown header' => {
        uri     => 'http://foo?bar=1',
        options => {
            headers => {
                'Host'    => 'foo.bar',
                'Foo-Bar' => 'foobar'
            }
        },
        params => {':get' => 1,},
        error  => 'Bad config for headers: Foo-Bar'
    },

    'POST - simple with form-urlencoded params' => {
        uri    => 'http://foo?bar=1',
        params => {
            ':post' => 1,
            foo     => 'bar',
            bar     => 'foo'
        },
        expect => q[POST http://foo?bar=1
Content-Length: 15
Content-Type: application/x-www-form-urlencoded

bar=foo&foo=bar
]
    },

    'POST - with white list headers' => {
        uri     => 'http://foo?bar=1',
        options => {headers => {'Host' => 'foo.bar',}},
        params  => {
            ':post'         => 1,
            ':content_type' => 'application/json',
            foo             => 'bar',
            bar             => 'foo'
        },
        expect => q[POST http://foo?bar=1
Host: foo.bar
Content-Length: 15
Content-Type: application/json

bar=foo&foo=bar
]
    },

    'POST - with unknown header' => {
        uri     => 'http://foo?bar=1',
        options => {
            headers => {
                'Host'    => 'foo.bar',
                'Foo-Bar' => 'foobar'
            }
        },
        params => {':post' => 1,},
        error  => 'Bad config for headers: Foo-Bar'
    },

    'POST - with data' => {
        uri    => 'http://foo?bar=1',
        params => {
            ':post'    => 1,
            ':content' => q[some multiline
text including JSON
{"foo":"bar"}]
        },
        expect => q[POST http://foo?bar=1
Content-Length: 48
Content-Type: application/x-www-form-urlencoded

some multiline
text including JSON
{"foo":"bar"}
]
    },
    'POST - with form-urlencoded hash data' => {
        uri    => 'http://foo?bar=1',
        params => {
            ':post'    => 1,
            ':content' => {
                foo => 'bar',
                bar => 'foo'
            }
        },
        expect => q[POST http://foo?bar=1
Content-Length: 15
Content-Type: application/x-www-form-urlencoded

bar=foo&foo=bar
]
    },
    'POST - with form-urlencoded arrray data' => {
        uri    => 'http://foo?bar=1',
        params => {
            ':post'    => 1,
            ':content' => [qw(foo bar bar foo)]
        },
        expect => q[POST http://foo?bar=1
Content-Length: 15
Content-Type: application/x-www-form-urlencoded

bar=foo&foo=bar
]
    },
    'POST - with form-urlencoded data with header' => {
        uri    => 'http://foo?bar=1',
        params => {
            ':post'         => 1,
            ':content'      => [qw(foo bar bar foo)],
            ':content_type' => 'application/json',
        },
        expect => q[POST http://foo?bar=1
Content-Length: 15
Content-Type: application/json

bar=foo&foo=bar
]
    },
};

run_tests(
    sub {
        my ($app) = @_;

        # Убираем вывод DEBUG
        my $logger = Log::Log4perl->get_logger('')->level($ERROR);

        my $LWPRequest = undef;
        {
            no warnings 'redefine';
            no strict 'refs';

            *{'LWP::UserAgent::request'} = sub {
                my ($mock, $request) = @_;

                $LWPRequest = $request;

                my $r = HTTP::Response->new(200);
                $r->request($request);
                $r->content('foo-content');
                return $r;
            };
        }

        my $self = $app->api_adfox;
        foreach my $testname (sort keys %$fixture) {
            my $test_data = $fixture->{$testname};

            $self->set_option($_, $test_data->{options}->{$_}) for keys %{$test_data->{options} // {}};

            my $err_msg = '';
            try {
                QBit::Application::Model::API::HTTP::get($self, $test_data->{uri}, %{$test_data->{params}});
            }
            catch {
                my ($exception) = @_;
                $err_msg = $exception->message;
            };

            eq_or_diff($err_msg, $test_data->{error}, "$testname. got error")
              if $err_msg || defined $test_data->{error};

            my $got = $LWPRequest ? $LWPRequest->as_string : '';
            $got =~ s/foo=bar&bar=foo/bar=foo&foo=bar/;
            eq_or_diff($got, $test_data->{expect}, "$testname") if defined $test_data->{expect};

            # teardown
            $LWPRequest = undef;
            $self->set_option($_, undef) for keys %{$test_data->{options} // {}};
        }
    },
    dont_create_database => 1,
);
