use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::MockObject::Extends;
use Test::More;
use Exception::Balance::IncorrectAnswer;
use Test::Differences;

sub get_correct_data_for_create_person {
    my %h = (
        operator_uid => 1000,
        client_id    => 123,
        person_id    => -1,
        type         => 'sw_yt',
        name         => 'name',
        phone        => 'phone',
        postaddress  => 'postaddress',
        postcode     => 'postcode',
        email        => 'aa@yandex.ru',
    );

    return %h;
}

run_tests(
    sub {
        my ($app) = @_;

        $app->api_balance;
        $app->{'api_balance'} = Test::MockObject::Extends->new($app->{'api_balance'});

        $app->api_balance->mock(
            'call',
            sub {
                my ($self, $method, $operator_uid, $opts) = @_;

                is($method, 'Balance2.CreatePerson', 'call is working with Balance2.CreatePerson');
                is($operator_uid, 1000, 'expected operator_uid in call');

                my $person_id = 6690573;
                return [$person_id];
            }
        );

        my $person_id = $app->api_balance->create_person(
            operator_uid => 1000,
            client_id    => 123,
            person_id    => -1,
            type         => 'sw_yt',
            name         => 'name',
            phone        => 'phone',
            postaddress  => 'postaddress',
            postcode     => 'postcode',
            email        => 'aa@yandex.ru',
        );

        is($person_id, 6690573, 'create_person() return expected person_id');

        $app->api_balance->mock(
            'call',
            sub {
                my ($self, $method, $operator_uid, $opts) = @_;

                my $func     = 'Balance2.CreatePerson';
                my $dump_str = '[
   1000,
   {
      "client_id" : 123,
      "email" : "aa@yandex.ru",
      "operator_uid" : 1000,
      "person_id" : -1,
      "phone" : "phone",
      "postaddress" : "postaddress",
      "postcode" : "postcode",
      "type" : "sw_yt"
   }
]
';

                my $error = "
<error>
  <msg>Missing mandatory person field 'name' for person type sw_yt</msg>
  <field>name</field>
  <wo-rollback>0</wo-rollback>
  <person-type>sw_yt</person-type>
  <method>Balance2.CreatePerson</method>
  <code>MISSING_MANDATORY_PERSON_FIELD</code>
  <parent-codes>
    <code>INVALID_PARAM</code>
    <code>EXCEPTION</code>
  </parent-codes>
  <contents>Missing mandatory person field 'name' for person type sw_yt</contents>
</error>
";

                throw Exception::Balance::IncorrectAnswer
                  sprintf("Got error from method \"%s\"\n opts: %s\n%s", $func, $dump_str, $error);
            }
        );

        eval {
            my $person_id = $app->api_balance->create_person(
                operator_uid => 1000,
                client_id    => 123,
                person_id    => -1,
                type         => 'sw_yt',
                # не указываем параметр 'name'
                phone       => 'phone',
                postaddress => 'postaddress',
                postcode    => 'postcode',
                email       => 'aa@yandex.ru',
            );
        };

        is(ref($@), 'Exception::Balance::IncorrectAnswer', 'Got expected Exception::Balance::IncorrectAnswer');

        my $tests = [
            {
                missing        => [qw(operator_uid)],
                expected_error => '[
   {
      "messages" : [
         "Key \"operator_uid\" required"
      ],
      "name" : [
         "operator_uid"
      ]
   }
]
',
            },
            {
                missing        => [qw(person_id)],
                expected_error => '[
   {
      "messages" : [
         "Key \"person_id\" required"
      ],
      "name" : [
         "person_id"
      ]
   }
]
',
            },
            {
                missing        => [qw(client_id)],
                expected_error => '[
   {
      "messages" : [
         "Key \"client_id\" required"
      ],
      "name" : [
         "client_id"
      ]
   }
]
',
            },
        ];

        foreach my $t (@{$tests}) {
            eval {
                my %opts = get_correct_data_for_create_person();
                foreach my $field (@{$t->{missing}}) {
                    delete $opts{$field};
                }

                my $person_id = $app->api_balance->create_person(%opts);
            };
            is(ref($@), 'Exception::Validator::Errors', 'Got expected Exception::Validator::Errors');
            eq_or_diff($@->message(), $t->{expected_error}, 'Got expected error message');
        }

    },
    init => [qw(api_balance)],
);
