use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::More;
use Test::Deep;

sub check_bik {
    my ($app) = @_;

    $app->api_balance->mock(
        'call',
        sub {
            my ($self, $method, $opts) = @_;

            is($method, 'Balance2.GetBank', 'running Balance2.GetBank');

            cmp_deeply($opts, {Bik => '044525700',}, 'got expected params',);

            return [
                {
                    bank_address => undef,
                    bank_city    => 'МОСКВА',
                    bank_name    => 'АО "РАЙФФАЙЗЕНБАНК"',
                    bik          => '044525700',
                    city         => 'МОСКВА',
                    cor_acc      => '30101810200000000700',
                    corr_account => '30101810200000000700',
                    hidden       => 0,
                    id           => 1120,
                    info         => undef,
                    name         => 'АО "РАЙФФАЙЗЕНБАНК"',
                    swift        => 'RZBMRUMMXXX',
                    update_dt    => '20180622T04:00:38',
                },
            ];

        }
    );

    my $data = $app->api_balance->get_bank(Bik => '044525700',);

    is_deeply(
        $data,
        [
            {
                bank_address => undef,
                bank_city    => 'МОСКВА',
                bank_name    => 'АО "РАЙФФАЙЗЕНБАНК"',
                bik          => '044525700',
                city         => 'МОСКВА',
                cor_acc      => '30101810200000000700',
                corr_account => '30101810200000000700',
                hidden       => 0,
                id           => 1120,
                info         => undef,
                name         => 'АО "РАЙФФАЙЗЕНБАНК"',
                swift        => 'RZBMRUMMXXX',
                update_dt    => '20180622T04:00:38',
            },
        ],
        'got expected answer',
    );

    return 1;
}

sub check_swift {
    my ($app) = @_;

    $app->api_balance->mock(
        'call',
        sub {
            my ($self, $method, $opts) = @_;

            is($method, 'Balance2.GetBank', 'running Balance2.GetBank');

            cmp_deeply($opts, {Swift => 'BOFAUS3NXXX',}, 'got expected params',);

            return [
                {
                    address => '222 BROADWAY',
                    bicint  => 'BOFAUS3NXXX',
                    country => 'UNITED STATES',
                    name    => 'BANK OF AMERICA, N.A.',
                    place   => 'NEW YORK,NY',
                    zipcode => 'NEW YORK,NY 10038',
                },
            ];

        }
    );

    my $data = $app->api_balance->get_bank(Swift => 'BOFAUS3NXXX',);

    is_deeply(
        $data,
        [
            {
                address => '222 BROADWAY',
                bicint  => 'BOFAUS3NXXX',
                country => 'UNITED STATES',
                name    => 'BANK OF AMERICA, N.A.',
                place   => 'NEW YORK,NY',
                zipcode => 'NEW YORK,NY 10038',
            },
        ],
        'got expected answer',
    );

    return 1;
}

sub check_incorrect_bik {
    my ($app) = @_;

    $app->api_balance->mock(
        'call',
        sub {
            my ($self, $method, $opts) = @_;

            is($method, 'Balance2.GetBank', 'running Balance2.GetBank');

            cmp_deeply($opts, {Bik => '123',}, 'got expected params',);

            throw Exception::Balance::IncorrectAnswer "Got error from method \"Balance2.GetBank\"
 opts: [
   {
      \"Bik\" : \"123\"
   }
]


<error>
  <msg>Object not found: Bank: primary keys: (u'123',)</msg>
  <object>Bank: primary keys: (u'123',)</object>
  <wo-rollback>0</wo-rollback>
  <method>Balance2.GetBank</method>
  <code>NOT_FOUND</code>
  <parent-codes>
    <code>EXCEPTION</code>
  </parent-codes>
  <contents>Object not found: Bank: primary keys: (u'123',)</contents>
</error>
";

        },
    );

    my $data;
    eval {$data = $app->api_balance->get_bank(Bik => '123',);};

    is(ref($@), 'Exception::Balance::NotFound', 'got expected exception');

    return 1;
}

sub check_incorrect_swift {
    my ($app) = @_;

    $app->api_balance->mock(
        'call',
        sub {
            my ($self, $method, $opts) = @_;

            is($method, 'Balance2.GetBank', 'running Balance2.GetBank');

            cmp_deeply($opts, {Swift => '123',}, 'got expected params',);

            throw Exception::Balance::IncorrectAnswer "Got error from method \"Balance2.GetBank\"
 opts: [
   {
      \"Swift\" : \"123\"
   }
]


<error>
  <msg>Bank with BIC 123XXXXXXXX not found in DB</msg>
  <object/>
  <wo-rollback>0</wo-rollback>
  <object-id>123XXXXXXXX</object-id>
  <method>Balance2.GetBank</method>
  <code>CLASSBANK_NOT_FOUND</code>
  <parent-codes>
    <code>NOT_FOUND</code>
    <code>EXCEPTION</code>
  </parent-codes>
  <contents>Bank with BIC 123XXXXXXXX not found in DB</contents>
</error>
";

        },
    );

    my $data;
    eval {$data = $app->api_balance->get_bank(Swift => '123',);};

    is(ref($@), 'Exception::Balance::NotFound', 'got expected exception');

    return 1;
}

run_tests(
    sub {
        my ($app) = @_;

        $app->api_balance;
        $app->{'api_balance'} = Test::MockObject::Extends->new($app->{'api_balance'});

        check_bik($app);
        check_swift($app);
        check_incorrect_bik($app);
        check_incorrect_swift($app);
    },
);
