#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Differences;
use Test::Exception;
use Test::More tests => 34;
use Test::Partner2::Simple;

my %data = (
    fields         => [qw(field1 field2 field3)],
    must_be_secure => 1,
    remote_addr    => 'remote_addr',
    server_name    => 'test.yandex.ru',
    session_id     => 'session_id',
    sessionid2     => 'sessionid2',
);

my $TVM_TICKET = 'mocked_ticket';

run_tests(
    sub {
        my ($app) = @_;

        my $bb = $app->api_blackbox;

        no warnings 'redefine';
        no strict 'refs';
        *{'Application::Model::API::Yandex::TVM::get_service_ticket'} = sub {
            return $TVM_TICKET;
        };
        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            fail 'Method get called while validation erorrs';
        };

        foreach my $param (qw(remote_addr server_name)) {
            throws_ok {
                $bb->sessionid(%data, $param => undef,);
            }
            'Exception::Validation::BadArguments', "Expected $param";
            eq_or_diff($@->message, "Expected $param", "Expected $param message");
        }

        throws_ok {
            $bb->sessionid(%data, extra => undef,);
        }
        'Exception::Validation::BadArguments', 'Extra params';
        eq_or_diff($@->message, 'Got unknown parameters: extra', 'Extra params message');

        throws_ok {
            $bb->sessionid(%data, session_id => undef,);
        }
        'Exception::BlackBox::NeedAuth', 'Not auth on master domain';
        eq_or_diff($@->message, 'No session', 'Not auth on master domain message');

        throws_ok {
            $bb->sessionid(
                %data,
                session_id  => undef,
                server_name => 'test.yandex.ua',
            );
        }
        'Exception::BlackBox::NeedAuth', 'Not auth on slave domain';
        eq_or_diff($@->message, 'No session', 'Not auth on slave domain message');

        throws_ok {
            $bb->sessionid(
                %data,
                session_id  => undef,
                server_name => 'test.yandex.ua',
            );
        }
        'Exception::BlackBox::NeedAuth', 'Not auth without MDA';
        eq_or_diff($@->message, 'No session', 'Not auth without MDA message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            my ($self, %fields) = @_;

            eq_or_diff \%fields,
              {
                emails       => 'getdefault',
                dbfields     => 'field1,field2,field3',
                host         => 'test.yandex.ru',
                method       => 'sessionid',
                sessionid    => 'session_id',
                sslsessionid => 'sessionid2',
                userip       => 'remote_addr',
              },
              'Params of method get';

            return {error => {content => 'Some known error',},};
        };
        throws_ok {
            $bb->sessionid(%data,);
        }
        'Exception::BlackBox::InternalError', 'Known blackbox error';
        eq_or_diff($@->message, 'Some known error', 'Known blackbox error message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {status => {},};
        };
        throws_ok {
            $bb->sessionid(%data,);
        }
        'Exception::BlackBox::InternalError', 'Unknown blackbox error';
        eq_or_diff($@->message, 'Unknown error', 'Unknown blackbox error message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {status => {content => '',},};
        };
        throws_ok {
            $bb->sessionid(%data,);
        }
        'Exception::BlackBox::NotSecure', 'Not secure connection while expected';
        eq_or_diff($@->message, 'Not secure connection', 'Not secure connection while expected message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth   => {secure => {content => 1,},},
                status => {
                    content => '',
                    id      => 5,
                },
            };
        };
        throws_ok {
            $bb->sessionid(%data,);
        }
        'Exception::BlackBox::NeedAuth', 'Not authorized';
        eq_or_diff($@->message, 'Not authorized', 'Not authorized message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth   => {secure => {content => 1,},},
                status => {
                    content => 'NEED_RESET',
                    id      => 0,
                },
            };
        };
        throws_ok {
            $bb->sessionid(%data,);
        }
        'Exception::BlackBox::NeedResign', 'Session wants to be fresh';
        eq_or_diff($@->message, 'Session wants to be fresh', 'Session wants to be fresh message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth   => {secure => {content => 1,},},
                status => {
                    content => '',
                    id      => 0,
                },
            };
        };
        throws_ok {
            $bb->sessionid(%data,);
        }
        'Exception::BlackBox::NeedAuth', 'Not authorized';
        eq_or_diff($@->message, 'Not authorized', 'Not authorized message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth   => {secure => {content => 1,},},
                status => {
                    content => 'NOAUTH',
                    id      => 0,
                },
            };
        };
        throws_ok {
            $bb->sessionid(%data, server_name => 'test.yandex.ua',);
        }
        'Exception::BlackBox::NeedAuth', 'Session too old';
        eq_or_diff($@->message, 'Session too old', 'Session too old message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth   => {secure => {content => 1,},},
                status => {
                    content => '',
                    id      => 0,
                },
            };
        };
        throws_ok {
            $bb->sessionid(%data, server_name => 'test.yandex.ua',);
        }
        'Exception::BlackBox::NeedResign', 'Session wants to be fresh or get from master domains';
        eq_or_diff(
            $@->message,
            'Session wants to be fresh or get from master domains',
            'Session wants to be fresh or get from master domains message'
        );

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth => {
                    secure => {content    => 1,},
                    social => {profile_id => {content => 1,},},
                },
                status => {
                    content => 'VALID',
                    id      => 0,
                },
            };
        };
        throws_ok {
            $bb->sessionid(%data, server_name => 'test.yandex.ua',);
        }
        'Exception::BlackBox::NeedRealLogin', 'Social auth';
        eq_or_diff($@->message, 'Yandex login - must have. Only social login doesn\'t matter', 'Social auth message');

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth => {
                    secure => {content    => 1,},
                    social => {profile_id => {content => 1,},},
                },
                login  => {content => 'some_login',},
                status => {
                    content => 'VALID',
                    id      => 0,
                },
            };
        };
        lives_ok {
            $bb->sessionid(%data, server_name => 'test.yandex.ua',);
        }
        'Social auth with postregistration';

        *{'QBit::Application::Model::API::Yandex::BlackBox::call'} = sub {
            return {
                auth   => {secure  => {content => 1,},},
                login  => {content => 'some_login',},
                status => {
                    content => 'VALID',
                    id      => 0,
                },
            };
        };
        lives_and {
            eq_or_diff(
                $bb->sessionid(%data,),
                {
                    auth => {secure => {content => 1,},},
                    from => {
                        host_passport  => 'passport.yandex.ru',
                        url_auth       => 'https://passport.yandex.ru/passport?mode=auth',
                        url_real_login => 'https://passport.yandex.ru/passport?mode=postregistration&create_login=1',
                        url_resign     => 'https://passport.yandex.ru/auth/update/?'
                    },
                    login  => {content => 'some_login',},
                    status => {
                        content => 'VALID',
                        id      => 0,
                    },
                }
            );
        }
        'Normal call';

    },
    dont_create_database => TRUE,
    init                 => [qw(api_blackbox api_tvm)],
    locale               => 'C',
);
