use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::More;
use Test::Deep;
use Test::Exception;

use JSON::XS qw(decode_json);
use URI;
use File::Temp qw(tempfile);
use File::Slurp qw(read_file write_file);
use Utils::TSV;

sub parse_response {
    my ($data) = @_;
    return parse_tsv(
        $data,
        headers => [
            qw(
              UpdateTime
              PageID
              DSPID
              PublisherID
              ContentID
              PublisherName
              ContentName
              Shows
              Wins
              Hits
              Price
              PartnerPrice
              CLID
              )
        ],
        string_escape => 1,
    );
}

sub call_ok {
    my ($app, %args) = @_;

    my %request_args = %args;
    delete $request_args{':content_file'};
    $request_args{$_} =~ s/-//g for (qw(startdate stopdate));
    $request_args{func} = 'get_publisher_stat';

    $args{pageid} //= 12345;

    my $reply = join(
        '',
        map {join("\t", values @$_) . "\n"} (
            ["$args{startdate} 00:00:00", $args{pageid}, 1, '', 41, '', 'Mocked content 1', 0, 0, 0, 0, 0, 0],
            ["$args{stopdate} 00:00:00",  $args{pageid}, 1, '', 42, '', 'Mocked content 2', 1, 3, 5, 8, 4, 0],
            ['#END'],
        ),
    );

    no warnings 'redefine';
    no strict 'refs';

    *{'LWP::UserAgent::request'} = sub {
        my ($mock, $request, $content_file_path) = @_;
        my $url = $request->uri;

        my %query_form = URI->new($url)->query_form();

        cmp_deeply([sort keys %query_form], ['data'], 'request keys',);

        cmp_deeply(decode_json($query_form{data}), {end_marker => 1, %request_args}, 'request values',);

        my $r = HTTP::Response->new(200);
        $r->request(HTTP::Request->new());
        if ($content_file_path) {
            $r->content('');
            write_file($content_file_path, $reply);
        } else {
            $r->content($reply);
        }
        return $r;
    };

    my $data = $app->api_http_bk->get_publisher_stat(%args);

    if (exists($args{':content_file'})) {
        $data = read_file($args{':content_file'}) or die "Failed to read $args{':content_file'}: $!";
        $data = parse_response($data);
        unlink $args{':content_file'} or die "Failed to unlink $args{':content_file'}: $!";
    }

    cmp_deeply($data, parse_response($reply), 'reply');
}

run_tests(
    sub {
        my ($app) = @_;

        call_ok(
            $app,
            startdate => '2017-04-01',
            stopdate  => '2017-04-03',
            pageid    => 42342,
        );

        call_ok(
            $app,
            startdate       => '2017-05-24',
            stopdate        => '2017-05-28',
            pageid          => 8976,
            ':content_file' => (tempfile(UNLINK => 1, OPEN => 0))[1],
        );
    },
);
