use strict;
use warnings FATAL => 'all';

use Test::More;
use Test::Differences qw(eq_or_diff);

use Test::Partner::Utils;
use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs mock_curdate);

use qbit;

BEGIN {$ENV{DIFF_OUTPUT_UNICODE} = 1}

my $tests = [
    {
        title  => 'good ping',
        method => 'ping',
        args   => [],
        data   => to_json({"response" => "Hello world!"}),
    },
    {
        title  => 'fail ping',
        method => 'ping',
        args   => [],
        data   => '{"response" => "Hello world!"}',
    },
    {
        title  => 'ok get_regions',
        method => 'get_regions',
        args   => [],
        data   => unset_utf(
            to_json(
                {
                    "response" => {
                        "regions" => [
                            {
                                "name"  => "Санкт-Петербург",
                                "oktmo" => "40000000"
                            },
                            {
                                "name"  => "Ленинградская область",
                                "oktmo" => "41000000"
                            }
                        ]
                    }
                }
            )
        )
    },
    {
        title  => 'short inn get_status',
        method => 'get_status',
        args   => [11111111],
        data   => unset_utf(
            to_json(
                {
                    'errorCode' => 'PLATFORM_ERROR_REQUEST_VALIDATION_ERROR',
                    'errorMessage' =>
"Ошибка платформы код:REQUEST_VALIDATION_ERROR, сообщение:ИНН должен содержать 12 символов (11111111)",
                    'errorXml' =>
"<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\"><soap:Body><GetMessageResponse xmlns=\"urn://x-artefacts-gnivc-ru/inplat/servin/OpenApiAsyncMessageConsumerService/types/1.0\"><ProcessingStatus>COMPLETED</ProcessingStatus><Message><SmzPlatformError xmlns=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersIntegrationService/types/1.0\" xmlns:S=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ns2=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersValidationService/types/1.0\"><Code>REQUEST_VALIDATION_ERROR</Code><Message>ИНН должен содержать 12 символов (11111111)</Message><Args><Key>INN</Key><Value>11111111</Value></Args></SmzPlatformError></Message></GetMessageResponse></soap:Body></soap:Envelope>",
                    'response' => undef
                }
            )
        ),
    },
    {
        title  => 'invalid inn get_status',
        method => 'get_status',
        args   => [111222333444],
        data   => unset_utf(
            to_json(
                {
                    'errorCode' => 'PLATFORM_ERROR_REQUEST_VALIDATION_ERROR',
                    'errorMessage' =>
"Ошибка платформы код:REQUEST_VALIDATION_ERROR, сообщение:Некорректный ИНН (111222333444)",
                    'errorXml' =>
"<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\"><soap:Body><GetMessageResponse xmlns=\"urn://x-artefacts-gnivc-ru/inplat/servin/OpenApiAsyncMessageConsumerService/types/1.0\"><ProcessingStatus>COMPLETED</ProcessingStatus><Message><SmzPlatformError xmlns=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersIntegrationService/types/1.0\" xmlns:S=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ns2=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersValidationService/types/1.0\"><Code>REQUEST_VALIDATION_ERROR</Code><Message>Некорректный ИНН (111222333444)</Message><Args><Key>INN</Key><Value>111111111111</Value></Args></SmzPlatformError></Message></GetMessageResponse></soap:Body></soap:Envelope>",
                    'response' => undef
                }
            )
        ),
    },
    {
        title  => 'not found inn get_status',
        method => 'get_status',
        args   => [123456789012],
        data   => unset_utf(
            to_json(
                {
                    'errorCode' => 'PLATFORM_ERROR_TAXPAYER_UNREGISTERED',
                    'errorMessage' =>
"Ошибка платформы код:TAXPAYER_UNREGISTERED, сообщение:Налогоплательщик не найден по ИНН 123456789012",
                    'errorXml' =>
"<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\"><soap:Body><GetMessageResponse xmlns=\"urn://x-artefacts-gnivc-ru/inplat/servin/OpenApiAsyncMessageConsumerService/types/1.0\"><ProcessingStatus>COMPLETED</ProcessingStatus><Message><SmzPlatformError xmlns=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersIntegrationService/types/1.0\" xmlns:S=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ns2=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersValidationService/types/1.0\"><Code>TAXPAYER_UNREGISTERED</Code><Message>Налогоплательщик не найден по ИНН 123456789012</Message><Args><Key>INN</Key><Value>575404565476</Value></Args></SmzPlatformError></Message></GetMessageResponse></soap:Body></soap:Envelope>",
                    'response' => undef
                }
            )
        ),
    },
    {
        title  => 'invalid inn bind_by_inn',
        method => 'bind_by_inn',
        args   => [111111111111],
        data   => unset_utf(
            to_json(
                {
                    'errorCode' => 'PLATFORM_ERROR_REQUEST_VALIDATION_ERROR',
                    'errorMessage' =>
"Ошибка платформы код:REQUEST_VALIDATION_ERROR, сообщение:Некорректный ИНН (111111111111)",
                    'errorXml' =>
"<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\"><soap:Body><GetMessageResponse xmlns=\"urn://x-artefacts-gnivc-ru/inplat/servin/OpenApiAsyncMessageConsumerService/types/1.0\"><ProcessingStatus>COMPLETED</ProcessingStatus><Message><SmzPlatformError xmlns=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersIntegrationService/types/1.0\" xmlns:S=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ns2=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersValidationService/types/1.0\"><Code>REQUEST_VALIDATION_ERROR</Code><Message>Некорректный ИНН (111111111111)</Message><Args><Key>INN</Key><Value>111111111111</Value></Args></SmzPlatformError></Message></GetMessageResponse></soap:Body></soap:Envelope>",
                    'response' => undef
                }
            )
        ),
    },
    {
        title  => 'request not found bind_status',
        method => 'bind_status',
        args   => [111111111111],
        data   => unset_utf(
            to_json(
                {
                    'errorCode' => 'PLATFORM_ERROR_REQUEST_VALIDATION_ERROR',
                    'errorMessage' =>
"Ошибка платформы код:REQUEST_VALIDATION_ERROR, сообщение:Запрос не найден",
                    'errorXml' =>
"<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\"><soap:Body><GetMessageResponse xmlns=\"urn://x-artefacts-gnivc-ru/inplat/servin/OpenApiAsyncMessageConsumerService/types/1.0\"><ProcessingStatus>COMPLETED</ProcessingStatus><Message><SmzPlatformError xmlns=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersIntegrationService/types/1.0\" xmlns:S=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ns2=\"urn://x-artefacts-gnivc-ru/ais3/SMZ/SmzPartnersValidationService/types/1.0\"><Code>REQUEST_VALIDATION_ERROR</Code><Message>Запрос не найден</Message><Args><Key>REQUEST_ID</Key><Value>111111111111</Value></Args></SmzPlatformError></Message></GetMessageResponse></soap:Body></soap:Envelope>",
                    'response' => undef
                }
            )
        ),
    },
    {
        title  => 'ok get_notifications',
        method => 'get_notifications',
        args   => [[123, 456]],
        data   => unset_utf(
            to_json(
                {
                    'response' => {
                        "innList" => [
                            {
                                "inn"           => 123,
                                "notifications" => [
                                    {
                                        "id"            => 10,
                                        "title"         => "Заголовок уведомления",
                                        "message"       => "Текст уведомления",
                                        "status"        => "Статус уведомления",
                                        "createdAt"     => "19T11:52:01.392Z",
                                        "updatedAt"     => "19T11:52:01.392Z",
                                        "partnerId"     => "1",
                                        "applicationId" => "2"
                                    }
                                ]
                            },
                            {
                                "inn"           => 456,
                                "notifications" => [
                                    {
                                        "id"            => 11,
                                        "title"         => "Заголовок уведомления",
                                        "message"       => "Текст уведомления",
                                        "status"        => "Статус уведомления",
                                        "createdAt"     => "19T11:52:01.392Z",
                                        "updatedAt"     => "19T11:52:01.392Z",
                                        "partnerId"     => "1",
                                        "applicationId" => "2"
                                    }
                                ]
                            }
                        ]
                    }
                }
            )
        ),
    },
    {
        title  => 'ok post_notif_read',
        method => 'post_notif_read',
        args   => [[{inn => 123, messageId => [10]}, {inn => 456, messageId => [11]}]],
        data => unset_utf(to_json({'response' => {"resultCode" => "OK"}})),
    },
];

my $test_data;

run_tests(
    sub {
        my ($app) = @_;

        my $result;
        mock_subs(
            {
                'QBit::Application::Model::API::HTTP::request' => sub {
                    my ($self, %params) = @_;
                    $params{content} = from_json($params{content}) if defined $params{content};
                    $result->{request} = \%params;
                    return $test_data;
                },
                'Application::Model::API::Yandex::TVM::get_service_ticket' => sub {
                    return "3:serv:CPlMEP";
                },
            }
        );

        mock_curdate("2020-07-17 10:10:00");
        my @result;
        for my $test (@$tests) {
            $result = {title => $test->{title}};
            my $method = $test->{method};
            $test_data = $test->{data};
            try {
                $result->{response} = $app->api_selfemployed_oebs->$method(@{$test->{args}});
            }
            catch {
                my ($e) = @_;
                $result->{error} = $e->message;
            };
            push @result, $result;
        }

        my $has = get_test_data_and_update_if_needed('data.json', \@result);
        eq_or_diff(from_json(to_json(\@result)), $has, 'body', {context => 5});
    },
    dont_create_database => 1,
    init                 => ['api_selfemployed_oebs', 'api_tvm'],
);
