#!/usr/bin/perl

use Test::Most;
use Test::More;

use qbit;

use TestAppDB;
use QBit::Application::Model::DB::mysql::Field;

my $app = TestAppDB->new();
$app->pre_run();

my $FIELDS_OPTS_BY_TYPE = {
    EMPTY => {
        generated => 'opts->>"$.field"',
        stored    => TRUE,
        not_null  => TRUE,
        default   => '0',
    },
    INT => {
        length        => 6,
        unsigned      => TRUE,
        zerofill      => TRUE,
        generated     => 'opts->>"$.field"',
        stored        => TRUE,
        not_null      => TRUE,
        autoincrement => TRUE,
        default       => 0,
    },
    FLOAT => {
        length    => 6,
        decimals  => 2,
        unsigned  => TRUE,
        zerofill  => TRUE,
        generated => 'opts->>"$.field"',
        stored    => TRUE,
        not_null  => TRUE,
        default   => 0,
    },
    BINARY => {
        length    => 6,
        generated => 'opts->>"$.field"',
        stored    => TRUE,
        not_null  => TRUE,
        default   => 0,
    },
    TIMESTAMP => {
        generated => 'opts->>"$.field"',
        stored    => TRUE,
        not_null  => TRUE,
        default   => 'CURRENT_TIMESTAMP',
    },
    TEXT => {
        binary    => TRUE,
        charset   => 'utf8',
        collate   => 'utf8_unicode_ci',
        generated => 'opts->>"$.field"',
        stored    => TRUE,
        not_null  => TRUE,
        default   => 0,
    },
    ENUM => {
        values    => ['one', 'two'],
        charset   => 'utf8',
        collate   => 'utf8_unicode_ci',
        generated => 'opts->>"$.field"',
        stored    => TRUE,
        not_null  => TRUE,
        default   => 0,
    },
    BLOB => {
        generated => 'opts->>"$.field"',
        stored    => TRUE,
        not_null  => TRUE,
    },
};

my $RESULTS = {
    'BIGINT' =>
'"field" BIGINT(6) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL AUTO_INCREMENT DEFAULT \'0\'',
    'BINARY'   => '"field" BINARY(6) GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'BIT'      => '"field" BIT(6) GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'BLOB'     => '"field" BLOB GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL',
    'BOOLEAN'  => '"field" BOOLEAN GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'CHAR'     => '"field" CHAR(255)',
    'DATE'     => '"field" DATE GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'DATETIME' => '"field" DATETIME GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'DECIMAL' =>
      '"field" DECIMAL(6, 2) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'ENUM' =>
'"field" ENUM(\'one\', \'two\') CHARACTER SET \'utf8\' GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'FLOAT' =>
      '"field" FLOAT(6, 2) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'INT' =>
'"field" INT(6) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL AUTO_INCREMENT DEFAULT \'0\'',
    'JSON'     => '"field" JSON GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL',
    'LONGBLOB' => '"field" LONGBLOB GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL',
    'LONGTEXT' =>
'"field" LONGTEXT BINARY CHARACTER SET \'utf8\' GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'MEDIUMBLOB' => '"field" MEDIUMBLOB GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL',
    'MEDIUMINT' =>
'"field" MEDIUMINT(6) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL AUTO_INCREMENT DEFAULT \'0\'',
    'MEDIUMTEXT' =>
'"field" MEDIUMTEXT BINARY CHARACTER SET \'utf8\' GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'NUMERIC' =>
      '"field" NUMERIC(6, 2) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'REAL' =>
      '"field" REAL(6, 2) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'SET' =>
'"field" SET(\'one\', \'two\') CHARACTER SET \'utf8\' GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'SMALLINT' =>
'"field" SMALLINT(6) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL AUTO_INCREMENT DEFAULT \'0\'',
    'TEXT' =>
      '"field" TEXT BINARY CHARACTER SET \'utf8\' GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'TIME'      => '"field" TIME GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'TIMESTAMP' => '"field" TIMESTAMP GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT CURRENT_TIMESTAMP',
    'TINYBLOB'  => '"field" TINYBLOB GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL',
    'TINYINT' =>
'"field" TINYINT(6) UNSIGNED ZEROFILL GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL AUTO_INCREMENT DEFAULT \'0\'',
    'TINYTEXT' =>
'"field" TINYTEXT BINARY CHARACTER SET \'utf8\' GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'VARBINARY' => '"field" VARBINARY(6) GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\'',
    'VARCHAR'   => '"field" VARCHAR(255)',
    'YEAR'      => '"field" YEAR GENERATED ALWAYS AS (opts->>"$.field") STORED NOT NULL DEFAULT \'0\''
};

foreach my $type (sort keys(%QBit::Application::Model::DB::mysql::Field::DATA_TYPES)) {
    $_ = QBit::Application::Model::DB::mysql::Field->new(
        db   => $app->db,
        name => 'field',
        type => $type,
        %{$FIELDS_OPTS_BY_TYPE->{$QBit::Application::Model::DB::mysql::Field::DATA_TYPES{$type}}}
    );

    is($_->create_sql(), $RESULTS->{$type}, $type);
}

$app->post_run();

done_testing();
