#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Most tests => 7;
use Test::Deep qw(cmp_deeply);
use Test::Differences qw(eq_or_diff);

use Test::Partner2::Mock qw(mock_subs);

use qbit;

use TestAppDB;

my $app = TestAppDB->new();

$app->pre_run();

$app->set_option(
    'locales' => {
        "en" => {"code" => "en_GB", "domain"  => "yandex.com", "name"   => "English"},
        "ru" => {"code" => "ru_RU", "default" => 1,            "domain" => "yandex.ru", "name" => "Русский"}
    }
);

$app->mysql_db;

my $SQL;
my $DATA;

mock_subs(
    {
        'QBit::Application::Model::DB::_do' => sub {
            shift;
            eq_or_diff(shift, $SQL, 'SQL');

            cmp_deeply(\@_, $DATA, 'Data');
          }
    }
);

my $t32 = 32;

my $TESTS_DATA = {
    'i18n all' => {
        data   => ['all', 'all'],
        filter => 13,
        fields => {field => 1, field_str => 'str', field_i18n => 'all'},
        sql    => q{UPDATE "table"
SET "field" = '1',
    "field_i18n_en" = ?,
    "field_i18n_ru" = ?,
    "field_str" = 'str'
WHERE (
    "table"."id" = '13'
)}
    },
    'i18n ru en' => {
        data   => ['en', 'ru'],
        filter => 13,
        fields => {field => 1, field_str => 'str', field_i18n => {ru => 'ru', en => 'en'}},
        sql    => q{UPDATE "table"
SET "field" = '1',
    "field_i18n_en" = ?,
    "field_i18n_ru" = ?,
    "field_str" = 'str'
WHERE (
    "table"."id" = '13'
)}
    },
    'i18n concat' => {
        data   => ['en', 'ru'],
        filter => 13,
        fields => {
            field => ['+', ['field', \1]],
            field_str => {CONCAT => ['field', \' ', 'field_str']},
            field_i18n => {ru => 'ru', en => 'en'}
        },
        sql => q{UPDATE "table"
SET "field" = ("table"."field" + '1'),
    "field_i18n_en" = ?,
    "field_i18n_ru" = ?,
    "field_str" = CONCAT("table"."field", ' ', "table"."field_str")
WHERE (
    "table"."id" = '13'
)}
    },
    'i18n - error' => {
        error  => 'You can not use expression for i18n fields in update',
        filter => 13,
        fields => {field_i18n => {CONCAT => ['field_i18n', \' all',]},},
        sql    => q{UPDATE "table"
SET "field" = ("table"."field" + '1'),
    "field_i18n_en" = ?,
    "field_i18n_ru" = ?,
    "field_str" = CONCAT("table"."field", ' ', "table"."field_str")
WHERE (
    "table"."id" = '13'
)}
    },
    'filter IN' => {
        data   => [],
        filter => ["field", 'IN', \[16, 256]],
        fields => {field => ['|', ['field', \32]],},
        sql    => q{UPDATE "table"
SET "field" = ("table"."field" | '32')
WHERE (
    "table"."field" IN ('16', '256')
)}
    },
    '7. filter OR' => {
        data   => [],
        filter => ['OR', [["field", 'IN', \[16, 256]], [['|', ['field', \1]], '!=', \1],]],
        fields => {field => ['|', ['field', \32]],},
        sql => q{UPDATE "table"
SET "field" = ("table"."field" | '32')
WHERE (
    "table"."field" IN ('16', '256')
    OR ("table"."field" | '1') <> '1'
)}
    },
    'filter error' => {
        data   => [],
        filter => {"field" => [16, 256]},
        fields => {field => 42,},
        sql    => q{UPDATE "table"
SET "field" = '42'
WHERE (
    "table"."field" IN ('16', '256')
)}
    },
};

foreach my $testname (sort keys %$TESTS_DATA) {
    (my $filter, my $fields, $SQL, $DATA, my $error) =
      @{$TESTS_DATA->{$testname}}{qw( filter fields  sql  data  error )};

    subtest 'check all elements in model' => sub {
        note sprintf('Start test "%s"', $testname);

        my $got_error = '';
        try {
            $filter = $app->db->filter($filter) if ref $filter;

            $app->mysql_db->table->edit($filter, $fields);
        }
        catch {
            $got_error = shift->message;
        };

        is($got_error, gettext($error // ''), 'error message');
    };
}

$app->post_run();
