#!/usr/bin/perl

use qbit;

use Test::More tests => 2 + 7;
use Test::Differences qw(eq_or_diff);
use Test::Exception;

use Test::Partner::Utils qw(get_test_data_and_update_if_needed);
use Test::Partner2::Mock qw(mock_curdate mock_subs restore_subs);
use Test::Partner2::Simple;

my $module = 'Cron::Methods::Moderation';
my $method = 'check_too_long_moderation';

my $need_error = FALSE;

# значальный пустой результат
my $empty_db = {
    crons => [],
    stat  => [],
};

# Список результатов
# Изначально одна запись
# Все остальные = первой + дополнение
my @expected_results = (
    {
        crons => [
            {
                id   => '1',
                name => 'moderation::check_too_long_moderation',
            },
        ],
        stat => [
            {    # first run
                cron_id  => '1',
                dt       => '2020-11-30 13:05:37',
                duration => '7',
                id       => '1',
                ignored  => '0',
                ok       => '1',
            },
        ],
    }
);

# Список дополнений к результатам
my @expected_bd_records = (
    {    # immediately run
        cron_id  => '1',
        dt       => '2020-11-30 13:05:37',
        duration => '7',
        id       => '2',
        ignored  => '1',
        ok       => '0',
    },
    {    # second run at almost end of period
        cron_id  => '1',
        dt       => '2020-11-30 13:59:59',
        duration => '7',
        id       => '3',
        ignored  => '1',
        ok       => '0',
    },
    {    # second run in next period
        cron_id  => '1',
        dt       => '2020-11-30 14:00:00',
        duration => '7',
        id       => '4',
        ignored  => '0',
        ok       => '1',
    },
    {    # third run in next period (with error)
        cron_id  => '1',
        dt       => '2020-11-30 15:01:23',
        duration => '7',
        id       => '5',
        ignored  => '0',
        ok       => '0',
    },
    {    # immediately fourth run after error
        cron_id  => '1',
        dt       => '2020-11-30 15:01:24',
        duration => '7',
        id       => '6',
        ignored  => '0',
        ok       => '1',
    },
);

# Итеративная генерация ожидаемых результатов
for (my $i = 0; $i < @expected_bd_records; $i++) {
    my $r = clone $expected_results[$i];
    push @{$r->{stat}}, $expected_bd_records[$i];
    push @expected_results, $r;
}

run_tests(
    sub {
        my ($cron) = @_;

        mock_cron_method();

        my $path       = $module->model_path();
        my $is_limited = FALSE;
        mock_subs(
            {
                'Application::get_time' => sub {return 7},
                'Cron::INFO'            => sub {return TRUE},
                'QBit::Cron::INFO'      => sub {return TRUE},
            }
        );

        subtest 'initial check' => sub {
            plan tests => 1;    # db

            eq_or_diff get_db($cron), $empty_db, 'db check';
        };

        subtest 'first run' => sub {
            plan tests => 1;    # db

            mock_curdate('2020-11-30 13:05:37');
            $is_limited = FALSE;
            $cron->do($path, $method);
            eq_or_diff get_db($cron), $expected_results[0], 'db check';
        };

        subtest 'immediately run' => sub {
            plan tests => 1;    # db

            $is_limited = TRUE;
            $cron->do($path, $method);
            eq_or_diff get_db($cron), $expected_results[1], 'db check';
        };

        subtest 'second run at almost end of period' => sub {
            plan tests => 1;    # db

            mock_curdate('2020-11-30 13:59:59');
            $is_limited = TRUE;
            $cron->do($path, $method);
            eq_or_diff get_db($cron), $expected_results[2], 'db check';
        };

        subtest 'second run in next period' => sub {
            plan tests => 1;    # db

            mock_curdate('2020-11-30 14:00:00');
            $is_limited = FALSE;
            $cron->do($path, $method);
            eq_or_diff get_db($cron), $expected_results[3], 'db check';
        };

        subtest 'third run in next period (with error)' => sub {
            plan tests => 2;    # exception + db

            mock_curdate('2020-11-30 15:01:23');
            $is_limited = FALSE;
            $need_error = TRUE;
            throws_ok {
                $cron->do($path, $method);
            }
            'Exception', 'Test exception';
            eq_or_diff get_db($cron), $expected_results[4], 'db check';
        };

        subtest 'immediately fourth run after error' => sub {
            plan tests => 1;    # db

            mock_curdate('2020-11-30 15:01:24');
            $is_limited = FALSE;
            $need_error = FALSE;
            $cron->do($path, $method);
            eq_or_diff get_db($cron), $expected_results[5], 'db check';
        };

    },
    application_package => 'Cron',
);

sub mock_cron_method {
    my $mock_sub = sub {
        throw 'Test exception' if $need_error;
        return TRUE;
    };
    my $full_method = $module . '::' . $method;
    my $original    = \&$full_method;
    mock_subs({$full_method => $mock_sub,});
    my $stash = package_stash($module);
    my $attrs = clone $stash->{'__CRON_ATTRS__'}{$module, $original};
    $attrs->{frequency_limit} = '1h';
    $stash->{'__CRON_ATTRS__'}{$module, $mock_sub} = $attrs;

    foreach (grep {$original eq $_->{sub}} @{$stash->{'__CRON__'}}) {
        $_->{attrs} = $attrs;
        $_->{sub}   = $mock_sub;
    }
}

sub get_db {
    my ($app) = @_;

    return {
        crons => $app->partner_db->crons->get_all(
            fields   => [qw(id name)],
            order_by => [['id']],
        ),
        stat => $app->partner_db->crons_raw_stat->get_all(
            fields   => [qw(id cron_id dt duration ignored ok)],
            order_by => [['id']],
        ),
    };
}
