#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::More tests => 14;
use Test::Deep;

use qbit;
use QBit::Validator;
use PiConstants qw($BLOCK_TYPE_LIMITS);

my $limit        = 5;
my $width        = 350;
my $block_limits = $BLOCK_TYPE_LIMITS->{adaptive};

my %data = (
    direct_block    => 'adaptive',
    limit           => $limit,
    adaptive_width  => $width,
    adaptive_height => 300,
);

ok(
    !QBit::Validator->new(
        data     => \%data,
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - ok'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_height => undef},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - undef'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_height => 'string'},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - string'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_height => --$block_limits->{'min_height'}},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - min value'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_height => $block_limits->{'max_height_coef'} * $limit + 1},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - max value'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_height => $block_limits->{'min_square'} / $width - 1},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - min square'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_height => ($block_limits->{'max_square_coef'} * $limit + 1) / $width},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - max square'
  );

ok(
    QBit::Validator->new(
        data     => {map {$_ => $data{$_}} qw(limit adaptive_width adaptive_height)},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {optional => TRUE},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type     => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - without direct_block'
  );

ok(
    QBit::Validator->new(
        data     => {map {$_ => $data{$_}} qw(direct_block adaptive_width adaptive_height)},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {optional => TRUE},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - without limit'
  );

ok(
    QBit::Validator->new(
        data     => {map {$_ => $data{$_}} qw(direct_block limit adaptive_height)},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {optional => TRUE},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - without adaptive_width'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_width => ''},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - with adaptive_width = undef'
  );

ok(
    QBit::Validator->new(
        data     => {%data, direct_block => '300x250'},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - direct_block="300x250"'
  );

ok(
    !QBit::Validator->new(
        data     => {%data, direct_block => '300x250', adaptive_height => ''},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - direct_block="300x250" and adaptive_height = undef'
  );

$data{'type'} = delete($data{'direct_block'});
ok(
    !QBit::Validator->new(
        data     => \%data,
        template => {
            type   => "hash",
            fields => {
                type            => {},
                limit           => {},
                adaptive_width  => {},
                adaptive_height => {type => 'adaptive_height'},
            }
        },
      )->has_errors,
    'adaptive_height - with type'
  );
