#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::More tests => 14;
use Test::Deep;

use qbit;
use QBit::Validator;
use PiConstants qw($BLOCK_TYPE_LIMITS);

my $limit        = 5;
my $height       = 300;
my $block_limits = $BLOCK_TYPE_LIMITS->{adaptive};

my %data = (
    direct_block    => 'adaptive',
    limit           => $limit,
    adaptive_width  => 350,
    adaptive_height => $height,
);

ok(
    !QBit::Validator->new(
        data     => \%data,
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - ok'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_width => undef},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - undef'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_width => 'string'},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - string'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_width => --$block_limits->{'min_width'}},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - min value'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_width => $block_limits->{'max_width_coef'} * $limit + 1},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - max value'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_width => $block_limits->{'min_square'} / $height - 1},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - min square'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_width => ($block_limits->{'max_square_coef'} * $limit + 1) / $height},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - max square'
  );

ok(
    QBit::Validator->new(
        data     => {map {$_ => $data{$_}} qw(limit adaptive_width adaptive_height)},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {optional => TRUE},
                limit           => {},
                adaptive_width  => {type     => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - without direct_block'
  );

ok(
    QBit::Validator->new(
        data     => {map {$_ => $data{$_}} qw(direct_block adaptive_width adaptive_height)},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {optional => TRUE},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - without limit'
  );

ok(
    QBit::Validator->new(
        data     => {map {$_ => $data{$_}} qw(direct_block limit adaptive_width)},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {optional => TRUE},
            }
        },
      )->has_errors,
    'adaptive_width - without adaptive_height'
  );

ok(
    QBit::Validator->new(
        data     => {%data, adaptive_height => ''},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - with adaptive_height = undef'
  );

ok(
    QBit::Validator->new(
        data     => {%data, direct_block => '300x250'},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - direct_block="300x250"'
  );

ok(
    !QBit::Validator->new(
        data     => {%data, direct_block => '300x250', adaptive_width => ''},
        template => {
            type   => "hash",
            fields => {
                direct_block    => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - direct_block="300x250" and adaptive_width = undef'
  );

$data{'type'} = delete($data{'direct_block'});
ok(
    !QBit::Validator->new(
        data     => \%data,
        template => {
            type   => "hash",
            fields => {
                type            => {},
                limit           => {},
                adaptive_width  => {type => 'adaptive_width'},
                adaptive_height => {},
            }
        },
      )->has_errors,
    'adaptive_width - with type'
  );
