#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::More tests => 28;

use qbit;
use QBit::Validator;
use QBit::Validator::Type::candidates;

use Application;

my $app = Application->new()->pre_run();

##############
# candidates #
##############

$QBit::Validator::Type::candidates::CANDIDATES_LISTS_MAX = 2;

my $TESTS = [
    {
        name   => 'undef',
        data   => undef,
        result => FALSE,
    },
    {
        name   => 'empty list',
        data   => [],
        result => TRUE,
    },
    {
        name   => 'host required',
        data   => [{}],
        result => FALSE,
    },
    {
        name   => 'host',
        data   => [{host => 'test.host.com'}],
        result => TRUE,
    },
    {
        name   => 'incorrect tags',
        data   => [{host => 'test.host.com', tags => ['test!']}],
        result => FALSE,
    },
    {
        name   => 'tags',
        data   => [{host => 'test.host.com', tags => ['test01']}],
        result => TRUE,
    },
    {
        name   => 'incorrect tags_blacklist',
        data   => [{host => 'test.host.com', tags_blacklist => ['test!']}],
        result => FALSE,
    },
    {
        name   => 'tags_blacklist',
        data   => [{host => 'test.host.com', tags_blacklist => ['test01']}],
        result => TRUE,
    },
    {
        name   => 'incorrect url_prefixes',
        data   => [{host => 'test.host.com', url_prefixes => ['test/url!']}],
        result => FALSE,
    },
    {
        name   => 'url_prefixes',
        data   => [{host => 'test.host.com', url_prefixes => ['/test/url']}],
        result => TRUE,
    },
    {
        name   => 'incorrect url_prefixes_blacklist',
        data   => [{host => 'test.host.com', url_prefixes_blacklist => ['test/url!']}],
        result => FALSE,
    },
    {
        name   => 'url_prefixes_blacklist',
        data   => [{host => 'test.host.com', url_prefixes_blacklist => ['/test/url']}],
        result => TRUE,
    },
    {
        name   => 'tags limit one domain',
        data   => [{host => 'test.host.com', tags => ['test01', 'test02']}],
        result => TRUE,
    },
    {
        name   => 'tags limit one domain - errors',
        data   => [{host => 'test.host.com', tags => ['test01', 'test02', 'test03']}],
        result => FALSE,
    },
    {
        name   => 'tags limit two domains',
        data   => [{host => 'test.host.com', tags => ['test01']}, {host => 'test.host.com', tags => ['test02']}],
        result => TRUE,
    },
    {
        name => 'tags limit two domains - error',
        data =>
          [{host => 'test.host.com', tags => ['test01']}, {host => 'test.host.com', tags => ['test02', 'test03']}],
        result => FALSE,
    },
    {
        name   => 'tags_blacklist limit one domain',
        data   => [{host => 'test.host.com', tags_blacklist => ['test01', 'test02']}],
        result => TRUE,
    },
    {
        name   => 'tags_blacklist limit one domain - errors',
        data   => [{host => 'test.host.com', tags_blacklist => ['test01', 'test02', 'test03']}],
        result => FALSE,
    },
    {
        name => 'tags_blacklist limit two domains',
        data => [
            {host => 'test.host.com', tags_blacklist => ['test01']},
            {host => 'test.host.com', tags_blacklist => ['test02']}
        ],
        result => TRUE,
    },
    {
        name => 'tags_blacklist limit two domains - error',
        data => [
            {host => 'test.host.com', tags_blacklist => ['test01']},
            {host => 'test.host.com', tags_blacklist => ['test02', 'test03']}
        ],
        result => FALSE,
    },
    {
        name   => 'url_prefixes limit one domain',
        data   => [{host => 'test.host.com', url_prefixes => ['/test/url', '/test/url2']}],
        result => TRUE,
    },
    {
        name   => 'url_prefixes limit one domain - errors',
        data   => [{host => 'test.host.com', url_prefixes => ['/test/url', '/test/url2', '/test/url3']}],
        result => FALSE,
    },
    {
        name => 'url_prefixes limit two domains',
        data => [
            {host => 'test.host.com', url_prefixes => ['/test/url']},
            {host => 'test.host.com', url_prefixes => ['/test/url2']}
        ],
        result => TRUE,
    },
    {
        name => 'url_prefixes limit two domains - error',
        data => [
            {host => 'test.host.com', url_prefixes => ['/test/url']},
            {host => 'test.host.com', url_prefixes => ['/test/url2', '/test/url3']}
        ],
        result => FALSE,
    },
    {
        name   => 'url_prefixes_blacklist limit one domain',
        data   => [{host => 'test.host.com', url_prefixes_blacklist => ['/test/url', '/test/url2']}],
        result => TRUE,
    },
    {
        name   => 'url_prefixes_blacklist limit one domain - errors',
        data   => [{host => 'test.host.com', url_prefixes_blacklist => ['/test/url', '/test/url2', '/test/url3']}],
        result => FALSE,
    },
    {
        name => 'url_prefixes_blacklist limit two domains',
        data => [
            {host => 'test.host.com', url_prefixes_blacklist => ['/test/url']},
            {host => 'test.host.com', url_prefixes_blacklist => ['/test/url2']}
        ],
        result => TRUE,
    },
    {
        name => 'url_prefixes_blacklist limit two domains - error',
        data => [
            {host => 'test.host.com', url_prefixes_blacklist => ['/test/url']},
            {host => 'test.host.com', url_prefixes_blacklist => ['/test/url2', '/test/url3']}
        ],
        result => FALSE,
    },
];

foreach (@$TESTS) {
    my $qv = QBit::Validator->new(data => $_->{'data'}, template => {type => 'candidates'}, app => $app);

    ok($_->{'result'} ? !$qv->has_errors : $qv->has_errors, $_->{'name'});
}
