#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Exception;
use Test::More tests => 39;
use Test::Deep;

use qbit;
use QBit::Validator;

########
# HASH #
########

ok(!QBit::Validator->new(data => {}, template => {type => "hash"},)->has_errors, 'Use constant type => "hash"');

ok(
    !QBit::Validator->new(data => undef, template => {type => "hash", optional => TRUE, fields => {key => {}}},)
      ->has_errors,
    'type => "hash" and optional'
  );

ok(QBit::Validator->new(data => [], template => {type => "hash"},)->has_errors, 'Data must be type "hash"');

ok(QBit::Validator->new(data => {}, template => {type => "hash", fields => {key => {}}},)->has_errors,
    'Check required keys');

ok(
    !QBit::Validator->new(data => {}, template => {type => "hash", fields => {key => {optional => TRUE}}},)->has_errors,
    'Check key with optional'
  );

ok(!QBit::Validator->new(data => {key => 1}, template => {type => "hash", fields => {key => {}}},)->has_errors,
    'Check key with scalar data');

ok(
    QBit::Validator->new(
        data     => {key  => 'abc'},
        template => {type => "hash", fields => {key => {regexp => qr/^\d+$/}}},
      )->has_errors,
    'Check key with scalar data and check data'
  );

ok(
    !QBit::Validator->new(
        data     => {key  => 'abc',  key2   => 5},
        template => {type => "hash", fields => {key => {regexp => qr/^[abc]{3}$/}, key2 => {eq => 5}}},
      )->has_errors,
    'Check two key with scalar data and check data'
  );

ok(
    QBit::Validator->new(
        data => {key => {key2 => 7}},
        template => {type => "hash", fields => {key => {type => "hash", fields => {key2 => {max => 4}}}}},
      )->has_errors,
    'Check key with hash data and check data'
  );

ok(
    QBit::Validator->new(data => {key => 1, key2 => 2}, template => {type => "hash", fields => {key => {}}},)
      ->has_errors,
    'Check extra keys'
  );

ok(
    !QBit::Validator->new(
        data     => {key  => 1,      key2  => 2},
        template => {type => "hash", extra => TRUE, fields => {key => {}}},
      )->has_errors,
    'Don\'t check extra keys with EXTRA'
  );

#
# check
#

my $error;
try {
    QBit::Validator->new(data => {key => 1}, template => {type => "hash", check => undef, fields => {key => {}}},);
}
catch {
    $error = TRUE;
};
ok($error, 'Option "check" must be code');

ok(
    !QBit::Validator->new(
        data     => {key => 2, key2 => 3, key3 => 5},
        template => {
            type  => "hash",
            check => sub {
                throw Exception::Validator::Fields gettext('Key3 must be equal key + key2')
                  if $_[1]->{'key3'} != $_[1]->{'key'} + $_[1]->{'key2'};
            },
            fields => {
                key  => {},
                key2 => {},
                key3 => {}
            }
        },
      )->has_errors,
    'Option "check" (no error)'
  );

ok(
    QBit::Validator->new(
        data     => {key => 2, key2 => 3, key3 => 7},
        template => {
            type  => "hash",
            check => sub {
                throw Exception::Validator::Fields gettext('Key3 must be equal key + key2')
                  if $_[1]->{'key3'} != $_[1]->{'key'} + $_[1]->{'key2'};
            },
            fields => {
                key  => {},
                key2 => {},
                key3 => {}
            }
        },
      )->has_errors,
    'Option "check" (error)'
  );

#
# deps
#

ok(
    !QBit::Validator->new(
        data     => {key => 1, key2 => 2,},
        template => {
            type   => "hash",
            fields => {
                key  => {},
                key2 => {},
            },
            deps => {key2 => 'key'},
        },
      )->has_errors,
    'Option "deps" (no error)'
  );

ok(
    QBit::Validator->new(
        data     => {key2 => 2,},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {},
            },
            deps => {key2 => 'key'},
        },
      )->has_errors,
    'Option "deps" (error)'
  );

#
# SKIP
#

ok(
    !QBit::Validator->new(
        data     => {key => 1, key2 => {key3 => 3, key4 => 4}},
        template => {
            type   => "hash",
            fields => {
                key  => {},
                key2 => {skip => TRUE},
            }
        },
      )->has_errors,
    'Use SKIP'
  );

#
#any_of (любой ключ из списка)
#

ok(
    !QBit::Validator->new(
        data     => {key2 => 2, key3 => 3},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
                key3 => {optional => TRUE}
            },
            any_of => ['key', 'key3']
        },
      )->has_errors,
    'Use any_of'
  );

ok(
    QBit::Validator->new(
        data     => {key2 => 2,},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
                key3 => {optional => TRUE}
            },
            any_of => ['key', 'key3']
        },
      )->has_errors,
    'Use any_of (error)'
  );

ok(
    QBit::Validator->new(
        data     => {},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
                key3 => {optional => TRUE}
            },
            any_of => ['key', 'key3']
        },
      )->has_errors,
    'Use any_of (error 2)'
  );

#
#one_of (ровно один ключ)
#

ok(
    !QBit::Validator->new(
        data     => {key2 => 2, key3 => 3},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
                key3 => {optional => TRUE}
            },
            one_of => ['key', 'key3']
        },
      )->has_errors,
    'Use one_of'
  );

ok(
    QBit::Validator->new(
        data     => {key2 => 2,},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
                key3 => {optional => TRUE}
            },
            one_of => ['key', 'key3']
        },
      )->has_errors,
    'Use one_of (error)'
  );

ok(
    QBit::Validator->new(
        data     => {key => 1, key2 => 2, key3 => 3},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
                key3 => {optional => TRUE}
            },
            one_of => ['key', 'key3']
        },
      )->has_errors,
    'Use one_of (error 2)'
  );

#
#all (все значения по ключам одного вида)
#

throws_ok {
    QBit::Validator->new(data => {}, template => {type => "hash", all => undef},);
}
'Exception::Validator', 'Option "all" must be type => "hash"';
is($@->message(), 'Option "all" must be HASH', 'error message');

throws_ok {
    QBit::Validator->new(
        data     => {},
        template => {
            all    => {},
            fields => {},
            type   => "hash",
        },
    );
}
'Exception::Validator', 'Option "all" vs option "fields"';
is($@->message(), 'Options "all" and "fields" can not be used together', 'error message');

ok(
    !QBit::Validator->new(
        data     => {key2 => 2, key3 => 3},
        template => {
            all  => {optional => TRUE,},
            type => "hash",
        },
      )->has_errors,
    'Use all'
  );

ok(
    !QBit::Validator->new(
        data     => {key2 => 2, key3 => 3},
        template => {
            all   => {optional => TRUE,},
            extra => FALSE,
            type  => "hash",
        },
      )->has_errors,
    'Use all with false extra option'
  );

ok(
    QBit::Validator->new(
        data     => {key2 => 2, key3 => 3},
        template => {
            all => {
                max      => 2,
                optional => TRUE,
            },
            type => "hash",
        },
      )->has_errors,
    'Use one_of (error)'
  );

#
#all_set (перечисленные поля либо все заданы(defined), либо undef)
#

$error = FALSE;
try {
    QBit::Validator->new(
        data     => {key  => 'key',  key2   => 'key2'},
        template => {type => "hash", fields => {key => {}, key2 => {}}, all_set => [['key', 'key3']]},
    );
}
catch {
    $error = shift->message();
};
is($error, gettext('Keys is not valid: %s', 'key3'), 'Option "all_set" check fields');

ok(
    !QBit::Validator->new(
        data     => {key => 'key', key2 => 'key'},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
            },
            all_set => [['key', 'key2']]
        },
      )->has_errors,
    'Use all_set'
  );

ok(
    QBit::Validator->new(
        data     => {key => 'key', key2 => undef},
        template => {
            type   => "hash",
            fields => {
                key  => {optional => TRUE},
                key2 => {optional => TRUE},
            },
            all_set => [['key', 'key2']]
        },
      )->has_errors,
    'Use all_set (error)'
  );

#
#must_exists (ключ должен присутствовать)
#

throws_ok(
    sub {
        QBit::Validator->new(
            data     => {key1 => undef},
            template => {
                type        => "hash",
                must_exists => [qw(key1 key2)],
                fields      => {
                    key1 => {optional => TRUE},
                    key2 => {optional => TRUE},
                },
            }
        );
    },
    qr/Exception::Validator: Expected key: key2/,
    'must_exists. absent key - fail'
);

ok(
    !QBit::Validator->new(
        data     => {key1 => 1, key2 => undef},
        template => {
            type        => "hash",
            must_exists => [qw(key1 key2)],
            fields      => {
                key1 => {optional => TRUE},
                key2 => {optional => TRUE},
                key3 => {optional => TRUE}
            },
        },
      )->has_errors,
    'must_exists. undef - correct'
  );

#
# check data to one error with deps
#

my $qv = QBit::Validator->new(
    data => {
        alt_width  => 0,
        alt_height => -1,
    },
    template => {
        type   => 'hash',
        fields => {
            alt_width  => {min => 1,},
            alt_height => {min => 1,},
        },
    }
);

ok($qv->has_errors, 'Check errors');

my @e = $qv->get_fields_with_error();

cmp_deeply(
    \@e,
    [
        {
            'path' => ['alt_height'],
            'msgs' => [gettext('Got value "%s" less then "%s"', -1, 1)]
        },
        {
            'path' => ['alt_width'],
            'msgs' => [gettext('Got value "%s" less then "%s"', 0, 1)]
        }
    ],
    'Two errors'
);

$qv = QBit::Validator->new(
    data => {
        alt_width  => 0,
        alt_height => -1,
    },
    template => {
        type   => 'hash',
        fields => {
            alt_width  => {min => 1,},
            alt_height => {min => 1,},
        },
        deps => {alt_width => 'alt_height'},
    }
);

ok($qv->has_errors, 'Check errors');

@e = $qv->get_fields_with_error();

cmp_deeply(
    \@e,
    [
        {
            'path' => ['alt_height'],
            'msgs' => [gettext('Got value "%s" less then "%s"', -1, 1)]
        }
    ],
    'One error'
);
